/*
 * OWL (OpenedHand Widget Library) Colour Swatch.
 *
 * Author: Rob Bradford  <rob@openedhand.com>
 *
 * Copyright (C) 2007 OpenedHand Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "owlcolourswatch.h"

#include <gtk/gtk.h>

G_DEFINE_TYPE (OwlColourSwatch, owl_colour_swatch, GTK_TYPE_FRAME);

#define COLOUR_SWATCH_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OWL_TYPE_COLOUR_SWATCH, OwlColourSwatchPrivate))

typedef struct _OwlColourSwatchPrivate OwlColourSwatchPrivate;

struct _OwlColourSwatchPrivate
{
  GtkWidget *draw_area;
  GdkPixbuf *pixbuf;
  GdkGC *gc;
  guint32 colour;
};

static void
owl_colour_swatch_get_property (GObject *object, guint property_id,
                              GValue *value, GParamSpec *pspec)
{
  switch (property_id) {
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
  }
}

static void
owl_colour_swatch_set_property (GObject *object, guint property_id,
                              const GValue *value, GParamSpec *pspec)
{
  switch (property_id) {
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
  }
}

static void
owl_colour_swatch_dispose (GObject *object)
{
  if (G_OBJECT_CLASS (owl_colour_swatch_parent_class)->dispose)
    G_OBJECT_CLASS (owl_colour_swatch_parent_class)->dispose (object);
}

static void
owl_colour_swatch_finalize (GObject *object)
{
  G_OBJECT_CLASS (owl_colour_swatch_parent_class)->finalize (object);
}

static void
update_pixbuf (OwlColourSwatch *swatch)
{
  OwlColourSwatchPrivate *priv = COLOUR_SWATCH_PRIVATE (swatch);

  gint width = priv->draw_area->allocation.width;
  gint height = priv->draw_area->allocation.height;

  if (priv->pixbuf != NULL)
    g_object_unref (priv->pixbuf);

  priv->pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, width, height);
  gdk_pixbuf_fill (priv->pixbuf, priv->colour);
}

static gint
expose_event (GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
  OwlColourSwatch *swatch = OWL_COLOUR_SWATCH (data);
  OwlColourSwatchPrivate *priv = COLOUR_SWATCH_PRIVATE (data);

  gint width = priv->draw_area->allocation.width;
  gint height = priv->draw_area->allocation.height;

  if (priv->pixbuf == NULL ||
      width != gdk_pixbuf_get_width (priv->pixbuf) ||
      height != gdk_pixbuf_get_height (priv->pixbuf))
    {
      update_pixbuf (swatch);
    }

  if (priv->gc == NULL)
    gdk_gc_new (widget->window);

  gdk_draw_pixbuf (widget->window,
                   priv->gc,
                   priv->pixbuf,
                   event->area.x - widget->allocation.x,
                   event->area.y - widget->allocation.y,
                   event->area.x,
                   event->area.y,
                   event->area.width,
                   event->area.height,
                   GDK_RGB_DITHER_MAX,
                   event->area.x - widget->allocation.x,
                   event->area.y - widget->allocation.y);


  return FALSE;
}


static void
owl_colour_swatch_class_init (OwlColourSwatchClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (OwlColourSwatchPrivate));

  object_class->get_property = owl_colour_swatch_get_property;
  object_class->set_property = owl_colour_swatch_set_property;
  object_class->dispose = owl_colour_swatch_dispose;
  object_class->finalize = owl_colour_swatch_finalize;
}

static void
owl_colour_swatch_init (OwlColourSwatch *self)
{
  OwlColourSwatchPrivate *priv = COLOUR_SWATCH_PRIVATE (self);

  priv->draw_area = gtk_alignment_new (0.5,0.5,0,0);

  priv->gc = NULL;
  priv->pixbuf = NULL;

  g_signal_connect (priv->draw_area, "expose-event",
                    G_CALLBACK (expose_event), self);

  priv->colour = 0xffffffff;

  gtk_widget_set_size_request (GTK_WIDGET (self), 24, 24);

  gtk_container_add (GTK_CONTAINER(self), priv->draw_area);

}

GtkWidget*
owl_colour_swatch_new (void)
{
  return GTK_WIDGET(g_object_new (OWL_TYPE_COLOUR_SWATCH, NULL));
}

void
owl_colour_swatch_set_colour (OwlColourSwatch *self, guint32 colour)
{
  OwlColourSwatchPrivate *priv = COLOUR_SWATCH_PRIVATE (self);

  priv->colour = colour;
  update_pixbuf (self);

  gtk_widget_queue_draw (priv->draw_area);
}

guint32
owl_colour_swatch_get_colour (OwlColourSwatch *self)
{
  OwlColourSwatchPrivate *priv = COLOUR_SWATCH_PRIVATE (self);

  return priv->colour;
}
