# Written by John Hoffman
# based on code by Uoti Urpala
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: BTcrypto.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Encrypted communication support.

@type KEY_LENGTH: C{int}
@var KEY_LENGTH: the length of the keys to generate
@type DH_PRIME: C{long}
@var DH_PRIME: a very large prime number
@type PAD_MAX: C{int}
@var PAD_MAX: the maximum amount of padding to add to the encryptes protocol header
@type DH_BYTES: C{int}
@var DH_BYTES: the number of bytes to use for key lengths

"""

from __future__ import generators   # for python 2.2
from random import randrange,randint,seed
try:
    from os import urandom
except:
    seed()
    urandom = lambda x: ''.join([chr(randint(0,255)) for i in xrange(x)])
from sha import sha

try:
    from Crypto.Cipher import ARC4
    CRYPTO_OK = True
except:
    CRYPTO_OK = False

KEY_LENGTH = 160
DH_PRIME = 0xFFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD129024E088A67CC74020BBEA63B139B22514A08798E3404DDEF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245E485B576625E7EC6F44C42E9A63A36210000000000090563
PAD_MAX = 200 # less than protocol maximum, and later assumed to be < 256
DH_BYTES = 96

def bytetonum(x):
    """Convert a long number in a string to a number.
    
    @type x: C{string}
    @param x: the data to convert
    @rtype: C{long}
    @return: the converted data
    
    """
    
    return long(x.encode('hex'), 16)

def numtobyte(x):
    """Convert a very large number to a string.
    
    @type x: C{long}
    @param x: the number to convert
    @rtype: C{string}
    @return: the string representation of the number
    
    """
    
    x = hex(x).lstrip('0x').rstrip('Ll')
    x = '0'*(192 - len(x)) + x
    return x.decode('hex')

class Crypto:
    """
    
    @type initiator: C{boolean}
    @ivar initiator: whether the connection was initiated locally
    @type disable_crypto: C{boolean}
    @ivar disable_crypto: whether crypto has been disabled
    @type privkey: C{long}
    @ivar privkey: randomly generated private key
    @type pubkey: C{string}
    @ivar pubkey: public key corresponding to the private key
    @type keylength: C{int}
    @ivar keylength: the key length to use
    @type _VC_pattern: C{string}
    @ivar _VC_pattern: the encrypted verification constant to expect from the peer
    @type S: C{string}
    @ivar S: the DH secret number, encoded as a byte array
    @type block3a: C{string}
    @ivar block3a: first encryption block
    @type block3bkey: C{string}
    @ivar block3bkey: key used to generate the second encryption block
    @type block3b: C{string}
    @ivar block3b: seconf encryption block
    @type encrypt: C{method}
    @ivar encrypt: the method to call to encrypt data
    @type decrypt: C{method}
    @ivar decrypt: the method to call to decrypt data
    @type _read: C{method}
    @ivar _read: the method to call to read decrypted data
    @type _write: C{method}
    @ivar _write: the method to call to write encrypted data
    
    """
    
    def __init__(self, initiator, disable_crypto = False):
        """Initialize the instance.
        
        @type initiator: C{boolean}
        @param initiator: whether the connection was initiated locally
        @type disable_crypto: C{boolean}
        @param disable_crypto: whether crypto has been disabled
            (optional, default is False)
        @raise NotImplementedError: if encryption is not installed
        
        """
        
        self.initiator = initiator
        self.disable_crypto = disable_crypto
        if not disable_crypto and not CRYPTO_OK:
            raise NotImplementedError, "attempt to run encryption w/ none installed"
        self.privkey = bytetonum(urandom(KEY_LENGTH/8))
        self.pubkey = numtobyte(pow(2, self.privkey, DH_PRIME))
        self.keylength = DH_BYTES
        self._VC_pattern = None

    def received_key(self, k):
        """Process a received key.
        
        @type k: C{string}
        @param k: the key that was received
        
        """
        
        self.S = numtobyte(pow(bytetonum(k), self.privkey, DH_PRIME))
        self.block3a = sha('req1'+self.S).digest()
        self.block3bkey = sha('req3'+self.S).digest()
        self.block3b = None

    def _gen_block3b(self, SKEY):
        """
        
        @type SKEY: C{string}
        @param SKEY: shared key (torrent info hash)
        
        """
        
        a = sha('req2'+SKEY).digest()
        return ''.join([ chr(ord(a[i])^ord(self.block3bkey[i]))
                         for i in xrange(20) ])

    def test_skey(self, s, SKEY):
        """Check that the encoding matches the encoded value.
        
        @type s: C{string}
        @param s: second received encryption block to verify
        @type SKEY: C{string}
        @param SKEY: shared key (torrent info hash)
        @rtype: C{boolean}
        @return: whether the encoding of SKEY matches s
        
        """
        
        block3b = self._gen_block3b(SKEY)
        if block3b != s:
            return False
        self.block3b = block3b
        if not self.disable_crypto:
            self.set_skey(SKEY)
        return True

    def set_skey(self, SKEY):
        """
        
        @type SKEY: C{string}
        @param SKEY: shared key (torrent info hash)
        
        """
        
        if not self.block3b:
            self.block3b = self._gen_block3b(SKEY)
        crypta = ARC4.new(sha('keyA'+self.S+SKEY).digest())
        cryptb = ARC4.new(sha('keyB'+self.S+SKEY).digest())
        if self.initiator:
            self.encrypt = crypta.encrypt
            self.decrypt = cryptb.decrypt
        else:
            self.encrypt = cryptb.encrypt
            self.decrypt = crypta.decrypt
        self.encrypt('x'*1024)  # discard first 1024 bytes
        self.decrypt('x'*1024)

    def VC_pattern(self):
        """Unknown.
        
        @rtype: C{string}
        @return: the encrypted verification constant to expect from the peer
        
        """
        if not self._VC_pattern:
            self._VC_pattern = self.decrypt('\x00'*8)
        return self._VC_pattern


    def read(self, s):
        """Decrypt and pass on the decrypted value.
        
        @type s: C{string}
        @param s: the string to decrypt
        
        """
        
        self._read(self.decrypt(s))

    def write(self, s):
        """Encrypt and pass on the encrypted value.
        
        @type s: C{string}
        @param s: the string to encrypt
        
        """
        
        self._write(self.encrypt(s))

    def setrawaccess(self, _read, _write):
        """Setup the methods to call for reading and writing.
        
        @type _read: C{method}
        @param _read: the method to call for reading decrypted data
        @type _write: C{method}
        @param _write: the method to call for writing encrypted data
        
        """
        
        self._read = _read
        self._write = _write

    def padding(self):
        """Generate a random amount of random padding.
        
        Generates random bytes, with a random length from 16 to L{PAD_MAX}.
        
        @rtype: C{string}
        @return: the randomly generated padding
        
        """
        
        return urandom(randrange(PAD_MAX-16)+16)
     
        
