# Written by John Hoffman
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: clock.py 71 2007-05-29 05:58:07Z camrdale-guest $

"""Implement wall clock time for Unix systems.

This module implements a clock() function that will return a non-decreasing
time, regardless of the system it is called on. This is necessary for Unix 
systems, whose clock() function instead returns the current processor time.

@type _MAXFORWARD: C{int}
@var _MAXFORWARD: the maximum number of seconds to allow the clock
    to move forward
@type _FUDGE: C{int}
@var _FUDGE: the fudged time change to use if the clock moved forward more
    than L{_MAXFORWARD}, or if the clock moved back
@type _RTIME: L{RelativeTime}
@var _RTIME: the RelativeTime instance to use

"""

from time import *
import sys

_MAXFORWARD = 100
_FUDGE = 1

class RelativeTime:
    """Calculate relative time on Unix systems.
    
    @type time: C{float}
    @ivar time: the last time value measured
    @type offset: C{float}
    @ivar offset: the offset to use from the current time values due to
        any changes made in the clock while the program was running
    
    """
    
    def __init__(self):
        """Initialize the time values."""
        self.time = time()
        self.offset = 0

    def get_time(self):        
        """Calculate a non-decreasing time.
        
        Uses the time() function to calculate non-decreasing time values.
        Checks to make sure the time values are non-decreasing, and also
        don't change by more than L{_MAXFORWARD} seconds within a reading.
        These could occur if the system clock was changed during the running
        of the program.
        
        @rtype: C{float}
        @return: the current time
        
        """
        
        t = time() + self.offset
        if t < self.time or t > self.time + _MAXFORWARD:
            self.time += _FUDGE
            self.offset += self.time - t
            return self.time
        self.time = t
        return t

if sys.platform != 'win32':
    _RTIME = RelativeTime()
    def clock():
        """Override the clock() function for Unix systems.
        
        This function will return a non-decreasing measure of the current
        time. This is only used on Unix systems. On Windows systems, the 
        clock() function from the C{time} module will be used.
        
        @rtype: C{float}
        @return: the relative time from the L{RelativeTime} instance
        
        """
        return _RTIME.get_time()