# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: selectpoll.py 191 2007-07-30 02:26:41Z camrdale-guest $

"""Provides socket polling for operating systems that don't support it.

This class is similar to the C{select} module in the standard library, and
provides the needed functionality for operating systems that don't support
it in the C{select} module.

@type POLLIN: C{int}
@var POLLIN: There is data to read
@type POLLOUT: C{int}
@var POLLOUT: Ready for output: writing will not block
@type POLLERR: C{int}
@var POLLERR: Error condition of some sort
@type POLLHUP: C{int}
@var POLLHUP: Hung up

"""

from select import select, error
from time import sleep
from types import IntType
from bisect import bisect
POLLIN = 1
POLLOUT = 2
POLLERR = 8
POLLHUP = 16

class poll:
    """Polling object to poll open sockets for events.
    
    @type rlist: C{list} of C{int}
    @ivar rlist: the socket file descriptors waiting for data
    @type wlist: C{list} of C{int}
    @ivar wlist: the socket file descriptors waiting to send data
    
    """
    
    def __init__(self):
        """Initialize the instance."""
        self.rlist = []
        self.wlist = []
        
    def register(self, f, t):
        """Register a socket for polling.
        
        @type f: C{int} or C{socket.socket}
        @param f: the file descriptor, or the socket to get the file descriptor from
        @type t: C{int}
        @param t: the events to poll for
        
        """
        
        if type(f) != IntType:
            f = f.fileno()
        if (t & POLLIN):
            insert(self.rlist, f)
        else:
            remove(self.rlist, f)
        if (t & POLLOUT):
            insert(self.wlist, f)
        else:
            remove(self.wlist, f)

    def unregister(self, f):
        """Unregister a socket from being polled.
        
        @type f: C{int} or C{socket.socket}
        @param f: the file descriptor, or the socket to get the file descriptor from
        
        """
        
        if type(f) != IntType:
            f = f.fileno()
        remove(self.rlist, f)
        remove(self.wlist, f)

    def poll(self, timeout = None):
        """Poll the registered sockets for events.
        
        @type timeout: C{int}
        @param timeout: number of seconds to wait before timing out and returning
        @rtype: C{list} of (C{int}, C{int})
        @return: the socket file descriptors and event types that have occurred on them
        
        """
        
        if self.rlist or self.wlist:
            try:
                r, w, e = select(self.rlist, self.wlist, [], timeout)
            except ValueError:
                return None
        else:
            sleep(timeout)
            return []
        result = []
        for s in r:
            result.append((s, POLLIN))
        for s in w:
            result.append((s, POLLOUT))
        return result

def remove(list, item):
    """Efficiently remove items from a sorted list.

    If the item is not in the liost, does nothing.
    
    @type list: C{list} of C{int}
    @param list: the list
    @type item: C{int}
    @param item: the item to remove
    
    """
    
    i = bisect(list, item)
    if i > 0 and list[i-1] == item:
        del list[i-1]

def insert(list, item):
    """Efficiently insert items into a sorted list.
    
    If the item is already in the list, does nothing.
    
    @type list: C{list} of C{int}
    @param list: the list
    @type item: C{int}
    @param item: the item to insert
    
    """
    
    i = bisect(list, item)
    if i == 0 or list[i-1] != item:
        list.insert(i, item)

def test_remove():
    """Test the remove function."""
    x = [2, 4, 6]
    remove(x, 2)
    assert x == [4, 6]
    x = [2, 4, 6]
    remove(x, 4)
    assert x == [2, 6]
    x = [2, 4, 6]
    remove(x, 6)
    assert x == [2, 4]
    x = [2, 4, 6]
    remove(x, 5)
    assert x == [2, 4, 6]
    x = [2, 4, 6]
    remove(x, 1)
    assert x == [2, 4, 6]
    x = [2, 4, 6]
    remove(x, 7)
    assert x == [2, 4, 6]
    x = [2, 4, 6]
    remove(x, 5)
    assert x == [2, 4, 6]
    x = []
    remove(x, 3)
    assert x == []

def test_insert():
    """Test the insert function."""
    x = [2, 4]
    insert(x, 1)
    assert x == [1, 2, 4]
    x = [2, 4]
    insert(x, 3)
    assert x == [2, 3, 4]
    x = [2, 4]
    insert(x, 5)
    assert x == [2, 4, 5]
    x = [2, 4]
    insert(x, 2)
    assert x == [2, 4]
    x = [2, 4]
    insert(x, 4)
    assert x == [2, 4]
    x = [2, 3, 4]
    insert(x, 3)
    assert x == [2, 3, 4]
    x = []
    insert(x, 3)
    assert x == [3]
