/*
 This file is part of DepQBF.

 DepQBF, a solver for quantified boolean formulae (QBF).	
 Copyright 2010 Florian Lonsing, Johannes Kepler University, Linz, Austria.

 See also http://fmv.jku.at/depqbf for further information.

 DepQBF is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or (at
 your option) any later version.

 DepQBF is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with DepQBF.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include <ctype.h>
#include "qdpll.h"
#include "qdpll_mem.h"
#include "qdpll_pcnf.h"
#include "qdpll_exit.h"
#include "qdpll_stack.h"
#include "qdpll_dep_man_generic.h"
#include "qdpll_dep_man_qdag.h"
#include "qdpll_dep_man_qdag_types.h"
#include "qdpll_config.h"


#define QDPLL_ABORT_QDPLL(cond,msg)				\
  do {								\
    if (cond)							\
      {								\
        fprintf (stderr, "[QDPLL] %s: %s\n", __func__, msg);	\
	fflush (stderr);					\
	abort();						\
      }								\
  } while (0)

enum QDPLLSolverState
{
  QDPLL_SOLVER_STATE_UNDEF = 0,
  QDPLL_SOLVER_STATE_SAT = 10,
  QDPLL_SOLVER_STATE_UNSAT = 20
};

typedef enum QDPLLSolverState QDPLLSolverState;

struct QDPLL
{
  QDPLLMemMan *mm;		/* Memory manager. */
  QDPLLDepManGeneric *dm;	/* Dependency manager. */
  LitIDStack add_stack;
  LitIDStack add_stack_tmp;
  QDPLLPCNF pcnf;
  unsigned int next_clause_id;
  unsigned int num_deps_init;

  /* Priority queue holding variable IDs for decision making. */
  unsigned int size_var_pqueue;
  unsigned int cnt_var_pqueue;
  VarID *var_pqueue;

  /* Stacks storing exist./univ. lits of working reason for faster
     type-reduce. */
  VarPtrStack wreason_a;
  VarPtrStack wreason_e;
  unsigned int dep_check_cnt;

  VarPtrStack choose_var_aclasses;
  VarPtrStack choose_var_eclasses;

  VarPtrStack res_cands;

  VarIDStack sdcl_assigned_vars;
  unsigned int sdcl_uncovered_clauses;
  /* The decision stack, also used as 'queue' for BCP. */
  VarID *assigned_vars;
  VarID *assigned_vars_top;
  VarID *assigned_vars_end;
  /* Region between 'bcp_ptr' and 'top': BCP queue. */
  VarID *bcp_ptr;
  VarID *old_bcp_ptr;

  /* Pointer to satisfied cube or empty clause. */
  Constraint *result_constraint;

  struct
  {
    unsigned int scope_opened:1;
    unsigned int found_empty_clause:1;
    unsigned int decision_level;
    unsigned int num_decisions;
    unsigned int num_backtracks;
    unsigned int pending_inits:1;
    unsigned int deps_init_trail_disabled;
    unsigned int lclauses_size;
    unsigned int lcubes_size;
    unsigned int clause_resizes;
    unsigned int cube_resizes;

    unsigned int restarting:1;
    unsigned int num_restarts;
    unsigned int num_irestarts;
    unsigned int last_backtracks;
    unsigned int num_restart_resets;
    unsigned int irestart_dist;
    unsigned int orestart_dist;
    double var_act_inc;
    struct
    {
      Var *var;
      QDPLLAssignment assignment;
      QDPLLVarMode mode;
      Constraint *antecedent;
    } forced_assignment;
    unsigned int abort_learning:1;
    int exceeded_soft_max_space;
    unsigned int disabled_clauses;
  } state;

  struct
  {
    unsigned int verbosity;
    unsigned int max_dec;
    unsigned int max_space;
    unsigned int soft_max_space;
  } options;
};

#define LEARN_VAR_NEG_MARK(v) ((v)->mark_learn0 = 1)
#define LEARN_VAR_NEG_UNMARK(v) ((v)->mark_learn0 = 0)
#define LEARN_VAR_NEG_MARKED(v) ((v)->mark_learn0)
#define LEARN_VAR_POS_MARK(v) ((v)->mark_learn1 = 1)
#define LEARN_VAR_POS_UNMARK(v) ((v)->mark_learn1 = 0)
#define LEARN_VAR_POS_MARKED(v) ((v)->mark_learn1)
#define LEARN_VAR_MARKED(v) ((v)->mark_learn0 || (v)->mark_learn1)
#define LEARN_VAR_UNMARK(v) ((v)->mark_learn0 = (v)->mark_learn1 = 0)

#define LINK_LAST(anchor,element,link)		\
  do {						\
    assert (!(element)->link.prev);		\
    assert (!(element)->link.next);		\
    if ((anchor).last)				\
      {						\
	assert (!(anchor).last->link.next);	\
	assert ((anchor).first);		\
	assert (!(anchor).first->link.prev);	\
	(anchor).last->link.next = (element);	\
      }						\
    else					\
      {						\
	assert (!(anchor).first);		\
	(anchor).first = (element);		\
      }						\
    (element)->link.prev = (anchor).last;	\
    (anchor).last = (element);			\
    (anchor).cnt++;				\
  } while (0)

#define LINK_FIRST(anchor,element,link)		\
  do {						\
    assert (!(element)->link.prev);		\
    assert (!(element)->link.next);		\
    (element)->link.next = (anchor).first;	\
    if ((anchor).first)				\
      {							\
	assert ((anchor).last);				\
	(anchor).first->link.prev = (element);		\
      }							\
    else						\
      {							\
	assert (!(anchor).last);			\
	(anchor).last = (element);			\
      }							\
    (anchor).first = (element);				\
    (anchor).cnt++;					\
  } while (0)

#define UNLINK(anchor,element,link)				\
  do {								\
    assert ((anchor).cnt);					\
    if ((element)->link.prev)					\
      {								\
	assert ((anchor).first);				\
	assert ((anchor).last);					\
	assert ((element)->link.prev->link.next == (element));	\
	(element)->link.prev->link.next = (element)->link.next; \
      }								\
    else							\
      {								\
	assert ((anchor).first == (element));			\
	(anchor).first = (element)->link.next;			\
      }								\
    if ((element)->link.next)					\
      {								\
	assert ((anchor).first);				\
	assert ((anchor).last);					\
	assert ((element)->link.next->link.prev == (element));	\
	(element)->link.next->link.prev = (element)->link.prev; \
      }								\
    else							\
      {								\
	assert ((anchor).last == (element));			\
	(anchor).last = (element)->link.prev;			\
      }								\
    (element)->link.prev = (element)->link.next = 0;		\
    (anchor).cnt--;						\
  } while (0)


/* Occurrence list maintenance. */
static void
occ_link_last (LitID lit, OccList * anchor, Constraint * element,
	       unsigned int offset)
{
  assert (offset < element->num_lits);
  assert (element->lits[offset] == lit);
  OccLink *link = &(element->occ_link[offset]);
  assert (!link->prev);
  assert (!link->poffset);
  assert (!link->next);
  assert (!link->noffset);
  if (anchor->last)
    {
      assert (!anchor->last->occ_link[anchor->loffset].next);
      assert (!anchor->last->occ_link[anchor->loffset].noffset);
      assert (anchor->first);
      assert (!anchor->first->occ_link[anchor->foffset].prev);
      assert (!anchor->first->occ_link[anchor->foffset].poffset);
      assert (anchor->loffset < anchor->last->num_lits);
      assert (anchor->last->lits[anchor->loffset] == lit);
      OccLink *other = &(anchor->last->occ_link[anchor->loffset]);
      other->next = (element);
      other->noffset = (offset);
    }
  else
    {
      assert (!anchor->first);
      assert (!anchor->foffset);
      anchor->first = (element);
      anchor->foffset = (offset);
    }
  link->prev = anchor->last;
  link->poffset = anchor->loffset;
  anchor->last = (element);
  anchor->loffset = (offset);
  anchor->cnt++;
}

static void
occ_link_first (LitID lit, OccList * anchor, Constraint * element,
		unsigned int offset)
{
  assert (offset < element->num_lits);
  assert (element->lits[offset] == lit);
  OccLink *link = &(element->occ_link[offset]);
  assert (!link->prev);
  assert (!link->poffset);
  assert (!link->next);
  assert (!link->noffset);
  link->next = anchor->first;
  link->noffset = anchor->foffset;
  if (anchor->first)
    {
      assert (anchor->last);
      assert (!anchor->first->occ_link[anchor->foffset].prev);
      assert (!anchor->first->occ_link[anchor->foffset].poffset);
      assert (!anchor->last->occ_link[anchor->loffset].next);
      assert (!anchor->last->occ_link[anchor->loffset].noffset);
      OccLink *other = &(anchor->first->occ_link[anchor->foffset]);
      other->prev = (element);
      other->poffset = (offset);
    }
  else
    {
      assert (!anchor->last);
      anchor->last = (element);
      anchor->loffset = (offset);
    }
  anchor->first = (element);
  anchor->foffset = (offset);
  anchor->cnt++;
}


static void
occ_unlink (LitID lit, OccList * anchor,
	    Constraint * element, unsigned int offset)
{
  assert (anchor->cnt);
  assert (offset < element->num_lits);
  assert (element->lits[offset] == lit);
  OccLink *link = &(element->occ_link[offset]);
  if (link->prev)
    {
      assert (anchor->first);
      assert (anchor->last);
      assert (link->poffset < link->prev->num_lits);
      assert (link->prev->occ_link[link->poffset].next == (element));
      assert (link->prev->lits[link->poffset] == lit);
      OccLink *other = &(link->prev->occ_link[link->poffset]);
      other->next = link->next;
      other->noffset = link->noffset;
    }
  else
    {
      assert (anchor->first == (element));
      anchor->first = link->next;
      anchor->foffset = link->noffset;
    }
  if (link->next)
    {
      assert (anchor->first);
      assert (anchor->last);
      assert (link->noffset < link->next->num_lits);
      assert (link->next->occ_link[link->noffset].prev == (element));
      assert (link->next->lits[link->noffset] == lit);
      OccLink *other = &(link->next->occ_link[link->noffset]);
      other->prev = link->prev;
      other->poffset = link->poffset;
    }
  else
    {
      assert (anchor->last == (element));
      anchor->last = link->prev;
      anchor->loffset = link->poffset;
    }
  link->prev = link->next = 0;
  link->poffset = link->noffset = 0;
  anchor->cnt--;
}


#define INTERNAL_SORTING_SWAP(T,p,q)		\
  do {						\
    T tmp = *(q);				\
    *(q) = *(p);				\
    *(p) = tmp;					\
  } while (0)

#define INTERNAL_SORTING_CMPSWAP(Q, T,cmp,p,q)		\
  do {							\
    if ((cmp) (Q, *(p), *(q)) > 0)			\
      INTERNAL_SORTING_SWAP (T, p, q);			\
  } while(0)

#define INTERNAL_INSERTION_SORT(Q,T,cmp,a,n)				\
  do {									\
    T pivot;								\
    int l = 0, r = (n) - 1, i, j;					\
    for (i = r; i > l; i--)						\
      INTERNAL_SORTING_CMPSWAP (Q, T, cmp, (a) + i - 1, (a) + i);	\
    for (i = l + 2; i <= r; i++)					\
      {									\
	j = i;								\
	pivot = (a)[i];							\
	while ((cmp) (Q, pivot, (a)[j - 1]) < 0)			\
	  {								\
	    (a)[j] = (a)[j - 1];					\
	    j--;							\
	  }								\
	(a)[j] = pivot;							\
      }									\
  } while (0)

#ifdef NDEBUG
#define CHECK_SORTED(Q,cmp,a,n) do { } while(0)
#else
#define CHECK_SORTED(Q,cmp,a,n)				\
  do {							\
    int i;						\
    for (i = 0; i < (n) - 1; i++)			\
      assert ((cmp) (Q, (a)[i], (a)[i + 1]) <= 0);	\
} while(0)
#endif

#define QDPLL_SORT(Q,T,cmp,a,n)				\
  do {							\
    T * aa = (a);					\
  int nn = (n);						\
  INTERNAL_INSERTION_SORT (Q, T, cmp, aa, nn);		\
  CHECK_SORTED (Q, cmp, aa, nn);			\
  } while (0)


static int is_clause_empty (QDPLL * qdpll, Constraint * clause);

static int is_clause_satisfied (QDPLL * qdpll, Constraint * clause);

static int
has_variable_active_occs_in_clauses (QDPLL * qdpll, Var * var,
				     OccList * occ_clauses, int check_prop);

static void
push_assigned_variable (QDPLL * qdpll, Var * var, QDPLLAssignment assignment,
			QDPLLVarMode mode);


/* -------------------- START: ASSERTION-ONLY CODE -------------------- */

static int
find_in_assigned_vars (QDPLL * qdpll, VarID id)
{
  VarID *p;
  for (p = qdpll->assigned_vars; p < qdpll->assigned_vars_top; p++)
    if (*p == id)
      return 1;

  return 0;
}


static unsigned int
count_in_notify_clause_watcher_list (LitIDStack * notify_list, LitID id)
{
  unsigned int cnt = 0;
  LitID *p, *e;
  for (p = notify_list->start, e = notify_list->top; p < e; p++)
    {
      assert (*p != 0);
      if (*p == id)
	cnt++;
    }

  return cnt;
}


static unsigned int
offset_in_notify_clause_watcher_list (LitIDStack * notify_list, LitID id)
{
  LitID *p, *e;
  for (p = notify_list->start, e = notify_list->top; p < e; p++)
    {
      assert (*p != 0);
      if (*p == id)
	return p - notify_list->start;
    }

  return -1;
}


static unsigned int
offset_in_clause (Constraint * clause, LitID id)
{
  assert (!clause->is_cube);
  LitID *p, *e;
  for (p = clause->lits, e = p + clause->num_lits; p < e; p++)
    {
      assert (*p != 0);
      if (*p == id)
	return p - clause->lits;
    }

  return -1;
}


static unsigned int
count_in_notify_literal_watcher_list (ConstraintPtrStack * notify_list,
				      Constraint * c)
{
  unsigned int cnt = 0;
  Constraint **p, **e;
  for (p = notify_list->start, e = notify_list->top; p < e; p++)
    {
      assert (*p != 0);
      if (*p == c)
	cnt++;
    }

  return cnt;
}


static void
print_assigned_vars (QDPLL * qdpll)
{
  Var *vars = qdpll->pcnf.vars, *v;

  VarID *p, *e;
  for (p = qdpll->assigned_vars, e = qdpll->assigned_vars_top; p < e; p++)
    {
      assert (*p > 0);
      v = VARID2VARPTR (vars, *p);
      fprintf (stderr,
	       "id=%d, type=%c(%d), dlevel=%d, value=%d, mode=%d, prop=%d\n",
	       v->id, QDPLL_SCOPE_FORALL (v->scope) ? 'A' : 'E',
	       v->scope->nesting, v->decision_level, v->assignment, v->mode,
	       v->mark_propagated);
    }
}


static void
print_lits (QDPLL * qdpll, LitID * lits, unsigned int num)
{
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *e;
  for (p = lits, e = p + num; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      fprintf (stderr, "%c(%d)%d",
	       QDPLL_SCOPE_FORALL (var->scope) ? 'A' : 'E',
	       var->scope->nesting, *p);
      if (QDPLL_VAR_ASSIGNED (var))
	{
	  char mode_char = 'X';
	  if (var->mode == QDPLL_VARMODE_UNIT)
	    mode_char = 'U';
	  else if (var->mode == QDPLL_VARMODE_PURE)
	    mode_char = 'P';
	  else if (var->mode == QDPLL_VARMODE_LBRANCH)
	    mode_char = 'L';
	  else if (var->mode == QDPLL_VARMODE_RBRANCH)
	    mode_char = 'R';
	  else
	    assert (0);
	  fprintf (stderr, "(%c%c)@%d",
		   QDPLL_VAR_ASSIGNED_TRUE (var) ? 'T' : 'F', mode_char,
		   var->decision_level);
	}
      fprintf (stderr, " ");
    }
  fprintf (stderr, "\n");
}


static void
print_constraint (QDPLL * qdpll, Constraint * c)
{
  print_lits (qdpll, c->lits, c->num_lits);
}


static int
constraint_has_lit (Constraint * c, LitID lit)
{
  LitID *p, *e;
  for (p = c->lits, e = p + c->num_lits; p < e; p++)
    {
      if (*p == lit)
	return 1;
    }

  return 0;
}


#ifndef NDEBUG

static void
assert_full_prefix_integrity (QDPLL * qdpll)
{
  assert (qdpll->pcnf.scopes.first);
  assert (qdpll->pcnf.scopes.first->nesting == QDPLL_DEFAULT_SCOPE_NESTING);
  assert (QDPLL_SCOPE_EXISTS (qdpll->pcnf.scopes.first));

  Scope *s, *n;
  for (s = qdpll->pcnf.scopes.first; s; s = n)
    {
      n = s->link.next;
      assert (s->nesting != QDPLL_DEFAULT_SCOPE_NESTING
	      || QDPLL_SCOPE_EXISTS (s));
      assert (s->nesting == QDPLL_DEFAULT_SCOPE_NESTING
	      || QDPLL_COUNT_STACK (s->vars) != 0);
      if (n)
	{
	  assert (s->nesting == n->nesting - 1);
	  assert (s->type != n->type);
	}
      VarIDStack *scope_vars = &s->vars;
      VarID *p, *e, v;
      for (p = scope_vars->start, e = scope_vars->top; p < e; p++)
	{
	  v = *p;
	  Var *var = qdpll->pcnf.vars + v;
	  assert (var->scope == s);
	  assert (var->id == v);
	  assert (!QDPLL_VAR_ASSIGNED (var) || var->mode == QDPLL_VARMODE_UNIT
		  || var->mode == QDPLL_VARMODE_PURE);
	  assert (!QDPLL_VAR_MARKED (var));
	  assert (!QDPLL_VAR_MARKED_PROPAGATED (var));
	  assert (QDPLL_VAR_HAS_OCCS (var));

	  LitID lit = -var->id;
	  OccListIterator it;
	  OLITER_INIT (it, var->neg_occ_clauses.first,
		       var->neg_occ_clauses.foffset, lit);
	  while (OLITER_CUR (it))
	    {
	      assert (constraint_has_lit (OLITER_CUR (it), -v));
	      OLITER_NEXT (it, lit);
	    }

	  lit = var->id;
	  OLITER_INIT (it, var->pos_occ_clauses.first,
		       var->pos_occ_clauses.foffset, lit);
	  while (OLITER_CUR (it))
	    {
	      assert (constraint_has_lit (OLITER_CUR (it), v));
	      OLITER_NEXT (it, lit);
	    }
	}
    }
}


static int
occs_have_constraint (LitID lit, OccList * occ_list, Constraint * c)
{
  OccListIterator it;
  OLITER_INIT (it, occ_list->first, occ_list->foffset, lit);
  while (OLITER_CUR (it))
    {
      if (OLITER_CUR (it) == c)
	return 1;
      OLITER_NEXT (it, lit);
    }
  return 0;
}


static unsigned int
count_occs (LitID lit, OccList * occs)
{
  unsigned int res = 0;
  OccListIterator it;
  OLITER_INIT (it, occs->first, occs->foffset, lit);
  while (OLITER_CUR (it))
    {
      res++;
      OLITER_NEXT (it, lit);
    }
  return res;
}


static void
assert_full_cnf_integrity_for_clauses (QDPLL * qdpll,
				       ConstraintList * clause_list)
{
  Constraint *c;
  for (c = clause_list->first; c; c = c->link.next)
    {
      assert (!c->is_cube);
      LitID *p1, *p2, *e, lit1, lit2;
      for (p1 = c->lits, e = p1 + c->num_lits; p1 < e; p1++)
	{
	  lit1 = *p1;
	  assert (lit1);
	  Var *v1 = LIT2VARPTR (qdpll->pcnf.vars, lit1);

	  assert (v1->neg_occ_clauses.cnt ==
		  count_occs (-v1->id, &v1->neg_occ_clauses));
	  assert (v1->pos_occ_clauses.cnt ==
		  count_occs (v1->id, &v1->pos_occ_clauses));

	  if (QDPLL_LIT_NEG (lit1))
	    assert (occs_have_constraint (lit1, &v1->neg_occ_clauses, c));
	  else
	    assert (occs_have_constraint (lit1, &v1->pos_occ_clauses, c));

	  for (p2 = p1 + 1; p2 < e; p2++)
	    {
	      lit2 = *p2;
	      assert (lit2 != lit1);
	      assert (lit2 != -lit1);
	      Var *v2 = LIT2VARPTR (qdpll->pcnf.vars, lit2);
	      assert (v1->scope->nesting <= v2->scope->nesting);
	    }
	}
    }
}


static unsigned int
count_constraints (ConstraintList * list)
{
  unsigned int res = 0;
  Constraint *c;
  for (c = list->first; c; c = c->link.next)
    res++;
  return res;
}


static void
assert_full_cnf_integrity (QDPLL * qdpll)
{
  assert (qdpll->pcnf.clauses.cnt ==
	  count_constraints (&(qdpll->pcnf.clauses)));
  assert_full_cnf_integrity_for_clauses (qdpll, &(qdpll->pcnf.clauses));
  assert_full_cnf_integrity_for_clauses (qdpll,
					 &(qdpll->pcnf.learnt_clauses));
}


static void
assert_full_formula_integrity (QDPLL * qdpll)
{
  assert_full_prefix_integrity (qdpll);
  assert_full_cnf_integrity (qdpll);
}


static void
assert_notify_lists_integrity_by_watcher (QDPLL * qdpll, LitID signed_id,
					  Constraint * watched_constraint)
{
  assert (watched_constraint->is_watched);
  LitID *p, *e;
  for (p = watched_constraint->lits, e = p + watched_constraint->num_lits;
       p < e; p++)
    {
      assert (*p != 0);
      Var *var = LIT2VARPTR (qdpll->pcnf.vars, *p);

      if (LIT2VARID (*p) == LIT2VARID (signed_id))
	continue;

      if (*p < 0)
	{
	  if (!watched_constraint->is_cube)
	    assert (count_in_notify_clause_watcher_list
		    (&(var->neg_notify_clause_watchers), signed_id) == 1);
	  else
	    assert (count_in_notify_clause_watcher_list
		    (&(var->pos_notify_clause_watchers), signed_id) == 1);

	}
      else
	{
	  if (!watched_constraint->is_cube)
	    assert (count_in_notify_clause_watcher_list
		    (&(var->pos_notify_clause_watchers), signed_id) == 1);
	  else
	    assert (count_in_notify_clause_watcher_list
		    (&(var->neg_notify_clause_watchers), signed_id) == 1);
	}
    }
}


static int
has_variable_active_occs_in_cubes (QDPLL * qdpll, Var * var,
				   OccList * occ_cubes);


static void
assert_all_pure_literals_and_clause_watchers_integrity (QDPLL * qdpll)
{
  Var *vars = qdpll->pcnf.vars;

  Scope *s;
  for (s = qdpll->pcnf.scopes.first; s; s = s->link.next)
    {
      VarIDStack *scope_vars = &s->vars;
      VarID *p, *e;
      for (p = scope_vars->start, e = scope_vars->top; p < e; p++)
	{
	  assert (*p > 0 && *p < qdpll->pcnf.size_vars);
	  Var *var = VARID2VARPTR (vars, *p);

	  assert (!QDPLL_VAR_MARKED_PROPAGATED (var)
		  || QDPLL_VAR_ASSIGNED (var));
	  assert (QDPLL_VAR_ASSIGNED (var)
		  || !QDPLL_VAR_MARKED_PROPAGATED (var));

	  int has_active_neg_occs_in_clauses =
	    has_variable_active_occs_in_clauses (qdpll, var,
						 &(var->neg_occ_clauses), 0);
	  int has_active_pos_occs_in_clauses =
	    has_variable_active_occs_in_clauses (qdpll, var,
						 &(var->pos_occ_clauses), 0);
	  int has_active_neg_occs_in_cubes =
	    has_variable_active_occs_in_cubes (qdpll, var,
					       &(var->neg_occ_cubes));
	  int has_active_pos_occs_in_cubes =
	    has_variable_active_occs_in_cubes (qdpll, var,
					       &(var->pos_occ_cubes));

	  if (has_active_neg_occs_in_clauses
	      && !has_active_pos_occs_in_clauses
	      && !has_active_pos_occs_in_cubes)
	    {
	      /* Pure: only negative occurrences left. */
	      /* Variable must have been pushed, but not necessarily
	         propagated already. */
	      assert (find_in_assigned_vars (qdpll, var->id));
	      assert (QDPLL_VAR_ASSIGNED (var));
	      assert (!QDPLL_VAR_EXISTS (var)
		      || QDPLL_VAR_ASSIGNED_FALSE (var));
	      assert (!QDPLL_VAR_FORALL (var)
		      || QDPLL_VAR_ASSIGNED_TRUE (var));

	      /* Exactly one watcher must be satisfied. */
	      assert (!QDPLL_VAR_HAS_NEG_OCCS (var) ||
		      !is_clause_satisfied (qdpll,
					    var->neg_occ_clauses.first));
	      assert (!QDPLL_VAR_HAS_POS_OCCS (var)
		      || is_clause_satisfied (qdpll,
					      var->pos_occ_clauses.first));

	    }
	  else if (!has_active_neg_occs_in_clauses
		   && !has_active_neg_occs_in_cubes
		   && has_active_pos_occs_in_clauses)
	    {
	      /* Pure: only pos occurrences left. */
	      /* Variable must have been pushed, but not necessarily
	         propagated already. */
	      assert (find_in_assigned_vars (qdpll, var->id));
	      assert (QDPLL_VAR_ASSIGNED (var));
	      assert (!QDPLL_VAR_EXISTS (var)
		      || QDPLL_VAR_ASSIGNED_TRUE (var));
	      assert (!QDPLL_VAR_FORALL (var)
		      || QDPLL_VAR_ASSIGNED_FALSE (var));
	      /* Exactly one watcher must be satisfied. */
	      assert (!QDPLL_VAR_HAS_POS_OCCS (var) ||
		      !is_clause_satisfied (qdpll,
					    var->pos_occ_clauses.first));
	      assert (!QDPLL_VAR_HAS_NEG_OCCS (var)
		      || is_clause_satisfied (qdpll,
					      var->neg_occ_clauses.first));
	    }
	  else if (!has_active_neg_occs_in_clauses
		   && !has_active_pos_occs_in_clauses
		   && !has_active_pos_occs_in_cubes
		   && !has_active_neg_occs_in_cubes)
	    {
	      /* Eliminated: no occurrences left. */
	      assert (!QDPLL_VAR_HAS_POS_OCCS (var)
		      || (!QDPLL_VAR_MARKED_PROPAGATED (var)
			  || QDPLL_VAR_ASSIGNED_FALSE (var))
		      || is_clause_satisfied (qdpll,
					      var->pos_occ_clauses.first));
	      assert (!QDPLL_VAR_HAS_NEG_OCCS (var)
		      || (!QDPLL_VAR_MARKED_PROPAGATED (var)
			  || QDPLL_VAR_ASSIGNED_TRUE (var))
		      || is_clause_satisfied (qdpll,
					      var->neg_occ_clauses.first));
	    }
	  else
	    {
	      /* Neither pure nor eliminated: both types of occurrences left. */
	      assert (!QDPLL_VAR_MARKED_PROPAGATED (var));
	      assert (!QDPLL_VAR_HAS_POS_OCCS (var) ||
		      !is_clause_satisfied (qdpll,
					    var->pos_occ_clauses.first)
		      || has_active_pos_occs_in_cubes);
	      assert (!QDPLL_VAR_HAS_NEG_OCCS (var)
		      || !is_clause_satisfied (qdpll,
					       var->neg_occ_clauses.first)
		      || has_active_neg_occs_in_cubes);
	    }

	  /* Check notify lists wrt. watched clauses. */
	  if (QDPLL_VAR_HAS_NEG_OCCS (var)
	      && !(QDPLL_VAR_ASSIGNED (var) && var->decision_level == 0))
	    {
	      if (!var->mark_is_neg_watching_cube)
		assert_notify_lists_integrity_by_watcher (qdpll, -var->id,
							  var->
							  neg_occ_clauses.
							  first);
	      else
		assert_notify_lists_integrity_by_watcher (qdpll, -var->id,
							  var->neg_occ_cubes.
							  first);
	    }

	  if (QDPLL_VAR_HAS_POS_OCCS (var)
	      && !(QDPLL_VAR_ASSIGNED (var) && var->decision_level == 0))
	    {
	      if (!var->mark_is_pos_watching_cube)
		assert_notify_lists_integrity_by_watcher (qdpll, var->id,
							  var->
							  pos_occ_clauses.
							  first);
	      else
		assert_notify_lists_integrity_by_watcher (qdpll, var->id,
							  var->pos_occ_cubes.
							  first);
	    }
	}
    }
}


static int
is_constraint_decided (QDPLL * qdpll, Constraint * c)
{
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *e;
  for (p = c->lits, e = p + c->num_lits; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      assert (var->id);
      if (!c->is_cube)
	{
	  if ((QDPLL_VAR_ASSIGNED_TRUE (var) && QDPLL_LIT_POS (lit)) ||
	      (QDPLL_VAR_ASSIGNED_FALSE (var) && QDPLL_LIT_NEG (lit)))
	    return 1;
	}
      else
	{
	  if ((QDPLL_VAR_ASSIGNED_TRUE (var) && QDPLL_LIT_NEG (lit)) ||
	      (QDPLL_VAR_ASSIGNED_FALSE (var) && QDPLL_LIT_POS (lit)))
	    return 1;
	}
    }
  return 0;
}


static int has_constraint_spurious_pure_lit (QDPLL * qdpll, Constraint * c);

static void
  assert_all_unit_literals_and_literal_watchers_integrity_for_clauses
  (QDPLL * qdpll, ConstraintList * clause_list)
{
  Var *vars = qdpll->pcnf.vars;
  Constraint *c;
  for (c = clause_list->first; c; c = c->link.next)
    {
      const int is_cube = c->is_cube;
      if (c->num_lits < 2)
	continue;

      assert (c->num_lits < 2 || c->lwatcher_pos < c->rwatcher_pos);
      assert (c->rwatcher_pos < c->num_lits);
      assert (c->lwatcher_pos < c->num_lits);

      unsigned int lwpos = c->lwatcher_pos;
      unsigned int rwpos = c->rwatcher_pos;

      LitID *lits = c->lits;
      LitID rwlit = *(lits + rwpos);
      LitID lwlit = *(lits + lwpos);
      assert (rwlit != 0);
      assert (lwlit != 0);
      assert (c->num_lits < 2 || rwlit != lwlit);
      assert (-rwlit != lwlit);

      Var *rwvar = LIT2VARPTR (vars, rwlit);
      Var *lwvar = LIT2VARPTR (vars, lwlit);
      assert (is_cube || QDPLL_VAR_EXISTS (rwvar));
      assert (!is_cube || QDPLL_VAR_FORALL (rwvar));

      assert (!QDPLL_VAR_ASSIGNED (rwvar) || is_constraint_decided (qdpll, c)
	      || has_constraint_spurious_pure_lit (qdpll, c));
      assert (!QDPLL_VAR_ASSIGNED (lwvar) || is_constraint_decided (qdpll, c)
	      || has_constraint_spurious_pure_lit (qdpll, c));

      ConstraintPtrStack *notify_list;
      if (QDPLL_LIT_NEG (rwlit))
	{
	  if (!is_cube)
	    notify_list = &(rwvar->pos_notify_lit_watchers);
	  else
	    notify_list = &(rwvar->neg_notify_lit_watchers);

	  assert (count_in_notify_literal_watcher_list (notify_list, c) == 1);
	  assert (((unsigned int) QDPLL_COUNT_STACK (*notify_list)) == 0 ||
		  c->offset_in_notify_list[1] <
		  ((unsigned int) QDPLL_COUNT_STACK (*notify_list)));
	  assert (((unsigned int) QDPLL_COUNT_STACK (*notify_list)) == 0
		  || c == notify_list->start[c->offset_in_notify_list[1]]);
	}
      else
	{
	  if (!is_cube)
	    notify_list = &(rwvar->neg_notify_lit_watchers);
	  else
	    notify_list = &(rwvar->pos_notify_lit_watchers);

	  assert (count_in_notify_literal_watcher_list (notify_list, c) == 1);
	  assert (((unsigned int) QDPLL_COUNT_STACK (*notify_list)) == 0 ||
		  c->offset_in_notify_list[1] <
		  ((unsigned int) QDPLL_COUNT_STACK (*notify_list)));
	  assert (QDPLL_COUNT_STACK (*notify_list) == 0
		  || c == notify_list->start[c->offset_in_notify_list[1]]);
	}

      if (QDPLL_LIT_NEG (lwlit))
	{
	  if (!is_cube)
	    notify_list = &(lwvar->pos_notify_lit_watchers);
	  else
	    notify_list = &(lwvar->neg_notify_lit_watchers);

	  assert (count_in_notify_literal_watcher_list (notify_list, c) == 1);
	  assert (((unsigned int) QDPLL_COUNT_STACK (*notify_list)) == 0 ||
		  c->offset_in_notify_list[0] <
		  ((unsigned int) QDPLL_COUNT_STACK (*notify_list)));
	  assert (((unsigned int) QDPLL_COUNT_STACK (*notify_list)) == 0
		  || c == notify_list->start[c->offset_in_notify_list[0]]);
	}
      else
	{
	  if (!is_cube)
	    notify_list = &(lwvar->neg_notify_lit_watchers);
	  else
	    notify_list = &(lwvar->pos_notify_lit_watchers);

	  assert (count_in_notify_literal_watcher_list (notify_list, c) == 1);
	  assert (((unsigned int) QDPLL_COUNT_STACK (*notify_list)) == 0 ||
		  c->offset_in_notify_list[0] <
		  ((unsigned int) QDPLL_COUNT_STACK (*notify_list)));
	  assert (((unsigned int) QDPLL_COUNT_STACK (*notify_list)) == 0
		  || c == notify_list->start[c->offset_in_notify_list[0]]);
	}

      LitID *ip, *ie;
      for (ip = c->lits, ie = c->lits + c->num_lits; ip < ie; ip++)
	{
	  if ((unsigned int) (ip - c->lits) != lwpos
	      && (unsigned int) (ip - c->lits) != rwpos)
	    {
	      Var *other = LIT2VARPTR (vars, *ip);
	      assert (count_in_notify_literal_watcher_list
		      (&(other->pos_notify_lit_watchers), c) == 0);
	      assert (count_in_notify_literal_watcher_list
		      (&(other->neg_notify_lit_watchers), c) == 0);
	    }
	}
    }
}


static void
assert_all_unit_literals_and_literal_watchers_integrity (QDPLL * qdpll)
{
  assert_all_unit_literals_and_literal_watchers_integrity_for_clauses (qdpll,
								       &
								       (qdpll->
									pcnf.
									clauses));
  assert_all_unit_literals_and_literal_watchers_integrity_for_clauses (qdpll,
								       &
								       (qdpll->
									pcnf.
									learnt_clauses));
  assert_all_unit_literals_and_literal_watchers_integrity_for_clauses (qdpll,
								       &
								       (qdpll->
									pcnf.
									learnt_cubes));
}


static void
assert_candidates_on_pqueue (QDPLL * qdpll)
{
  QDPLLDepManGeneric *dm = qdpll->dm;
  Var *p, *e;
  for (p = qdpll->pcnf.vars, e = p + qdpll->pcnf.size_vars; p < e; p++)
    {
      /* All variables which are candidates must be either (already)
         assigned or must occur on priority queue. */
      if (p->id)
	assert (!dm->is_candidate (dm, p->id) || QDPLL_VAR_ASSIGNED (p)
		|| p->priority_pos != QDPLL_INVALID_PQUEUE_POS);
    }
}


static void
assert_learn_vars_unmarked (QDPLL * qdpll)
{
  Var *p, *e;
  for (p = qdpll->pcnf.vars, e = p + qdpll->pcnf.size_vars; p < e; p++)
    {
      assert (!p->mark_learn0);
      assert (!p->mark_learn1);
      assert (!p->mark_res_cand);
      assert (!QDPLL_VAR_POS_MARKED (p));
      assert (!QDPLL_VAR_NEG_MARKED (p));
      assert (!QDPLL_VAR_MARKED (p));
    }
}


/* This is for checking asserting clauses only. */
static int
assert_is_clause_satisfied_by_univ_lit (QDPLL * qdpll, LitID implied,
					Constraint * clause)
{
  assert (!clause->is_cube);
  int found_implied = 0;
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *e;
  for (p = clause->lits, e = p + clause->num_lits; p < e; p++)
    {
      LitID lit = *p;
      if (lit != implied)
	{
	  Var *var = LIT2VARPTR (vars, lit);
	  if ((QDPLL_LIT_NEG (lit) && QDPLL_VAR_ASSIGNED_FALSE (var)) ||
	      (QDPLL_LIT_POS (lit) && QDPLL_VAR_ASSIGNED_TRUE (var)))
	    {
	      if (!QDPLL_SCOPE_FORALL (var->scope))
		return 0;
	      if (!found_implied)
		return 0;
	      /* Clause must be satisfied by universal pure literal. */
	      if (!(var->mode == QDPLL_VARMODE_PURE))
		return 0;
	    }
	}
      else
	found_implied = 1;
    }
  return 1;
}


static int
clause_has_true_ignored_lit (QDPLL * qdpll, Constraint * c, VarID skip_varid,
			     unsigned int ignore_dlevel);


static void
assert_re_init_deps_disabled_clauses (QDPLL * qdpll)
{
  ConstraintList *clauses = &(qdpll->pcnf.clauses);
  Constraint *c;
  for (c = clauses->first; c; c = c->link.next)
    {
      assert (!c->disabled || clause_has_true_ignored_lit (qdpll, c, 0, 0));
      assert (!clause_has_true_ignored_lit (qdpll, c, 0, 0) || c->disabled);
    }
}


static void
assert_lits_sorted (QDPLL * qdpll, LitID * lit_start, LitID * lit_end)
{
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *prev, *e;
  for (prev = p = lit_start, e = lit_end; p < e; p++)
    {
      if (!*p)
	continue;
      Var *pvar = LIT2VARPTR (vars, *p);
      Var *prev_var = LIT2VARPTR (vars, *prev);
      assert (prev_var->scope->nesting <= pvar->scope->nesting);
      prev = p;
    }
}


static void
assert_lits_no_holes (QDPLL * qdpll, LitID * lit_start, LitID * lit_end)
{
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *e;
  for (p = lit_start, e = lit_end; p < e; p++)
    assert (*p);
}


static void
assert_pushed_pure_lits (QDPLL * qdpll)
{
  Var *vars = qdpll->pcnf.vars;
  VarID *p, *e;
  for (p = qdpll->assigned_vars, e = qdpll->assigned_vars_top; p < e; p++)
    {
      Var *var = VARID2VARPTR (vars, *p);
      if (var->mode == QDPLL_VARMODE_PURE)
	{
	  int has_neg_occ_clauses =
	    has_variable_active_occs_in_clauses (qdpll, var,
						 &(var->neg_occ_clauses), 0);
	  int has_pos_occ_clauses =
	    has_variable_active_occs_in_clauses (qdpll, var,
						 &(var->pos_occ_clauses), 0);
	  int has_pos_occ_cubes =
	    has_variable_active_occs_in_cubes (qdpll, var,
					       &(var->pos_occ_cubes));
	  int has_neg_occ_cubes =
	    has_variable_active_occs_in_cubes (qdpll, var,
					       &(var->neg_occ_cubes));
	  assert (!(has_neg_occ_clauses && has_pos_occ_clauses));
	  assert (!(has_neg_occ_cubes && has_pos_occ_cubes));
	  assert (!(has_neg_occ_clauses && has_pos_occ_cubes));
	  assert (!(has_pos_occ_clauses && has_neg_occ_cubes));
	  if (QDPLL_SCOPE_FORALL (var->scope))
	    {
	      if (var->assignment == QDPLL_ASSIGNMENT_FALSE)
		{
		  assert (!has_neg_occ_clauses && !has_neg_occ_cubes);
		}
	      else
		{
		  assert (var->assignment == QDPLL_ASSIGNMENT_TRUE);
		  assert (!has_pos_occ_clauses && !has_pos_occ_cubes);
		}
	    }
	  else
	    {
	      assert (QDPLL_SCOPE_EXISTS (var->scope));
	      if (var->assignment == QDPLL_ASSIGNMENT_FALSE)
		{
		  assert (!has_pos_occ_clauses && !has_pos_occ_cubes);
		}
	      else
		{
		  assert (var->assignment == QDPLL_ASSIGNMENT_TRUE);
		  assert (!has_neg_occ_clauses && !has_neg_occ_cubes);
		}
	    }
	}
    }
}

/* Check if current constraint contains complementary lits. */
static int
working_clause_is_tautologous (QDPLL * qdpll, LitIDStack * lit_stack,
			       const QDPLLQuantifierType type)
{
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *e;
  for (p = lit_stack->start, e = lit_stack->top; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      assert (LEARN_VAR_MARKED (var));

      if (LEARN_VAR_POS_MARKED (var) && LEARN_VAR_NEG_MARKED (var))
	{
	  assert (0);
	  assert (type != QDPLL_SCOPE_FORALL (var->scope));
	  return 1;
	}
    }

  return 0;
}

#endif
/* -------------------- END: ASSERTION-ONLY CODE -------------------- */


/* -------------------- START: VARIABLE PRIORITY-QUEUE -------------------- */

static void
var_pqueue_adjust (QDPLL * qdpll, unsigned int size)
{
  unsigned int old_size;

  if ((old_size = qdpll->size_var_pqueue) < size)
    {
      QDPLLMemMan *mm = qdpll->mm;
      qdpll->var_pqueue = qdpll_realloc (mm, qdpll->var_pqueue,
					 old_size * sizeof (VarID),
					 size * sizeof (VarID));
      qdpll->size_var_pqueue = size;
    }
}


static unsigned int
var_pqueue_get_left_child_pos (unsigned int cur_pos)
{
  assert (cur_pos != QDPLL_INVALID_PQUEUE_POS);
  return 2 * cur_pos + 1;
}


static unsigned int
var_pqueue_get_right_child_pos (unsigned int cur_pos)
{
  assert (cur_pos != QDPLL_INVALID_PQUEUE_POS);
  return 2 * (cur_pos + 1);
}


static unsigned int
var_pqueue_get_parent_pos (unsigned int cur_pos)
{
  assert (cur_pos != QDPLL_INVALID_PQUEUE_POS);
  unsigned int result;
  result = (cur_pos - 1) / 2;
  assert (cur_pos == var_pqueue_get_right_child_pos (result) ||
	  cur_pos == var_pqueue_get_left_child_pos (result));
  return result;
}


static int
var_pqueue_compare (QDPLL * qdpll, unsigned int pos_a, unsigned int pos_b)
{
  assert (pos_a != QDPLL_INVALID_PQUEUE_POS);
  assert (pos_b != QDPLL_INVALID_PQUEUE_POS);
  assert (pos_a < qdpll->cnt_var_pqueue);
  assert (pos_b < qdpll->cnt_var_pqueue);

  unsigned int *var_pqueue = qdpll->var_pqueue;
  Var *vars = qdpll->pcnf.vars;
  assert (*(var_pqueue + pos_a) > 0);
  assert (*(var_pqueue + pos_b) > 0);
  Var *var_a = VARID2VARPTR (vars, *(var_pqueue + pos_a));
  Var *var_b = VARID2VARPTR (vars, *(var_pqueue + pos_b));

  double var_a_priority = var_a->priority;
  double var_b_priority = var_b->priority;

  if (var_a_priority < var_b_priority)
    return -1;
  else if (var_a_priority == var_b_priority)
    return 0;
  else
    return 1;
}


static void
var_pqueue_swap (QDPLL * qdpll, unsigned int pos_a, unsigned int pos_b)
{
  assert (pos_a != pos_b);
  assert (pos_a != QDPLL_INVALID_PQUEUE_POS);
  assert (pos_b != QDPLL_INVALID_PQUEUE_POS);
  assert (pos_a < qdpll->cnt_var_pqueue);
  assert (pos_b < qdpll->cnt_var_pqueue);

  VarID *var_pqueue = qdpll->var_pqueue;
  unsigned int tmp, *ptr_a, *ptr_b;

  ptr_a = var_pqueue + pos_a;
  tmp = *ptr_a;
  ptr_b = var_pqueue + pos_b;

  Var *vars = qdpll->pcnf.vars;
  assert (*ptr_a > 0);
  assert (*ptr_b > 0);
  Var *var_a = VARID2VARPTR (vars, *ptr_a);
  Var *var_b = VARID2VARPTR (vars, *ptr_b);

  assert (var_a->priority_pos == pos_a);
  assert (var_b->priority_pos == pos_b);

  *ptr_a = *ptr_b;
  var_b->priority_pos = pos_a;

  *ptr_b = tmp;
  var_a->priority_pos = pos_b;
}


static void
var_pqueue_up_heap (QDPLL * qdpll, unsigned int cur_pos)
{
  assert (cur_pos != QDPLL_INVALID_PQUEUE_POS);
  assert (cur_pos < qdpll->cnt_var_pqueue);

  while (cur_pos > 0)
    {
      unsigned int parent_pos = var_pqueue_get_parent_pos (cur_pos);

      if (var_pqueue_compare (qdpll, cur_pos, parent_pos) <= 0)
	break;

      var_pqueue_swap (qdpll, cur_pos, parent_pos);
      cur_pos = parent_pos;
    }
}


static void
var_pqueue_down_heap (QDPLL * qdpll, unsigned int cur_pos)
{
  assert (cur_pos != QDPLL_INVALID_PQUEUE_POS);
  assert (cur_pos < qdpll->cnt_var_pqueue);

  unsigned int child_pos, left_child_pos, right_child_pos;
  unsigned int count = qdpll->cnt_var_pqueue;

  for (;;)
    {
      left_child_pos = var_pqueue_get_left_child_pos (cur_pos);

      if (left_child_pos >= count)
	break;			/* has no left child */

      right_child_pos = var_pqueue_get_right_child_pos (cur_pos);

      if (right_child_pos < count &&
	  var_pqueue_compare (qdpll, left_child_pos, right_child_pos) < 0)
	child_pos = right_child_pos;
      else
	child_pos = left_child_pos;

      if (var_pqueue_compare (qdpll, cur_pos, child_pos) < 0)
	{
	  var_pqueue_swap (qdpll, cur_pos, child_pos);
	  cur_pos = child_pos;
	}
      else
	break;
    }
}


static void
assert_var_pqueue_condition (QDPLL * qdpll)
{
  unsigned int *var_pqueue = qdpll->var_pqueue;
  unsigned int pos, no_children, left_child_pos, right_child_pos;
  Var *vars = qdpll->pcnf.vars;
  no_children = qdpll->cnt_var_pqueue / 2;

  for (pos = 0; pos < qdpll->cnt_var_pqueue; pos++)
    {
      unsigned int *cur, *left, *right;
      Var *cur_var, *left_var, *right_var;

      cur = var_pqueue + pos;
      assert (*cur > 0);
      cur_var = VARID2VARPTR (vars, *cur);
      assert (cur_var->priority_pos == pos);

      left_child_pos = var_pqueue_get_left_child_pos (pos);
      right_child_pos = var_pqueue_get_right_child_pos (pos);

      if (pos < no_children)
	{
	  assert (left_child_pos < qdpll->cnt_var_pqueue);

	  left = var_pqueue + left_child_pos;
	  assert (*left > 0);
	  left_var = VARID2VARPTR (vars, *left);
	  assert (left_var->priority_pos == left_child_pos);

	  if (right_child_pos < qdpll->cnt_var_pqueue)
	    {
	      right = var_pqueue + right_child_pos;
	      assert (*right > 0);
	      right_var = VARID2VARPTR (vars, *right);
	      assert (right_var->priority_pos == right_child_pos);
	    }

	  assert (cur_var->priority >= left_var->priority);
	  assert (right_child_pos >= qdpll->cnt_var_pqueue ||
		  cur_var->priority >= right_var->priority);
	}
      else			/* has no children */
	{
	  assert (right_child_pos >= qdpll->cnt_var_pqueue);
	  assert (left_child_pos >= qdpll->cnt_var_pqueue);
	}
    }
}


static void
var_pqueue_increase_key (QDPLL * qdpll, VarID id)
{
  unsigned int cur_pos = VARID2VARPTR (qdpll->pcnf.vars, id)->priority_pos;
  var_pqueue_up_heap (qdpll, cur_pos);
#ifndef NDEBUG
#if QDPLL_PQ_ASSERT_HEAP_CONDITION_INCREASE_KEY
  assert_var_pqueue_condition (qdpll);
#endif
#endif
}


static void
var_pqueue_insert (QDPLL * qdpll, VarID id, double priority)
{
  assert (id > 0);
  unsigned int pos, cnt = qdpll->cnt_var_pqueue, size =
    qdpll->size_var_pqueue;
  pos = cnt;

  if (cnt == size)
    var_pqueue_adjust (qdpll, size ? 2 * size : 1);

  qdpll->var_pqueue[pos] = id;
  Var *var = VARID2VARPTR (qdpll->pcnf.vars, id);
  assert (QDPLL_VAR_HAS_OCCS (var));
  var->priority = priority;
  assert (var->priority_pos == QDPLL_INVALID_PQUEUE_POS);
  var->priority_pos = pos;
  cnt++;
  qdpll->cnt_var_pqueue = cnt;

  var_pqueue_up_heap (qdpll, pos);

#ifndef NDEBUG
#if QDPLL_PQ_ASSERT_HEAP_CONDITION_INSERT
  assert_var_pqueue_condition (qdpll);
#endif
#endif
}


static VarID
var_pqueue_remove_first (QDPLL * qdpll)
{
  Var *vars = qdpll->pcnf.vars;
  VarID *var_pqueue = qdpll->var_pqueue;
  VarID last, result = 0;
  unsigned int cnt = qdpll->cnt_var_pqueue;

  if (cnt == 0)
    return result;

  result = var_pqueue[0];
  assert (result > 0);
  Var *last_var, *result_var = VARID2VARPTR (vars, result);

  cnt--;
  last = var_pqueue[cnt];
  last_var = VARID2VARPTR (vars, last);
  var_pqueue[0] = last;
  assert (last_var->priority_pos == cnt);
  last_var->priority_pos = 0;
  result_var->priority_pos = QDPLL_INVALID_PQUEUE_POS;

  qdpll->cnt_var_pqueue = cnt;

  return result;
}


static VarID
var_pqueue_remove_min (QDPLL * qdpll)
{
  VarID result = 0;

  if (qdpll->cnt_var_pqueue == 0)
    return result;

  result = var_pqueue_remove_first (qdpll);

  if (qdpll->cnt_var_pqueue > 0)
    var_pqueue_down_heap (qdpll, 0);

#ifndef NDEBUG
#if QDPLL_PQ_ASSERT_HEAP_CONDITION_REMOVE_MIN
  assert_var_pqueue_condition (qdpll);
#endif
#endif

  return result;
}


static VarID
var_pqueue_access_min (QDPLL * qdpll)
{
  VarID *var_pqueue = qdpll->var_pqueue;
  unsigned int cnt = qdpll->cnt_var_pqueue;

  if (cnt == 0)
    return 0;
  else
    {
      assert (var_pqueue[0] > 0);
      return var_pqueue[0];
    }
}


static VarID
var_pqueue_remove_elem (QDPLL * qdpll, unsigned int remove_pos)
{
  assert (remove_pos != QDPLL_INVALID_PQUEUE_POS);
  assert (remove_pos < qdpll->cnt_var_pqueue);

#ifndef NDEBUG
#if QDPLL_PQ_ASSERT_HEAP_CONDITION_REMOVE_ELEM
  assert_var_pqueue_condition (qdpll);
#endif
#endif

  VarID last_id, remove_id;
  unsigned int cnt = qdpll->cnt_var_pqueue;
  Var *last_var, *remove_var, *vars = qdpll->pcnf.vars;
  VarID *var_pqueue = qdpll->var_pqueue;
  VarID *remove_ptr = var_pqueue + remove_pos;

  remove_id = *remove_ptr;
  assert (remove_id > 0);
  remove_var = VARID2VARPTR (vars, remove_id);
  assert (remove_var->priority_pos == remove_pos);
  remove_var->priority_pos = QDPLL_INVALID_PQUEUE_POS;

  cnt--;
  last_id = var_pqueue[cnt];
  assert (last_id > 0);
  qdpll->cnt_var_pqueue = cnt;

  if (remove_pos != cnt)
    {
      *remove_ptr = last_id;
      last_var = VARID2VARPTR (vars, last_id);
      assert (last_var->priority_pos == cnt);
      last_var->priority_pos = remove_pos;
      var_pqueue_up_heap (qdpll, remove_pos);
      var_pqueue_down_heap (qdpll, remove_pos);
    }

#ifndef NDEBUG
#if QDPLL_PQ_ASSERT_HEAP_CONDITION_REMOVE_ELEM
  assert_var_pqueue_condition (qdpll);
#endif
#endif

  return remove_id;
}

/* -------------------- END: VARIABLE PRIORITY-QUEUE -------------------- */


static size_t
size_assigned_vars (QDPLL * qdpll)
{
  return qdpll->assigned_vars_end - qdpll->assigned_vars;
}


static size_t
count_assigned_vars (QDPLL * qdpll)
{
  return qdpll->assigned_vars_top - qdpll->assigned_vars;
}


static void
enlarge_assigned_vars (QDPLL * qdpll)
{
  size_t old_size = size_assigned_vars (qdpll);
  size_t old_count = count_assigned_vars (qdpll);
  assert (old_size == old_count);
  size_t old_bcp_index = qdpll->bcp_ptr - qdpll->assigned_vars;
  size_t old_old_bcp_index = qdpll->old_bcp_ptr - qdpll->assigned_vars;
  size_t new_size = old_size ? 2 * old_size : 1;
  qdpll->assigned_vars =
    (VarID *) qdpll_realloc (qdpll->mm, qdpll->assigned_vars,
			     old_size * sizeof (VarID),
			     new_size * sizeof (VarID));
  qdpll->assigned_vars_end = qdpll->assigned_vars + new_size;
  qdpll->assigned_vars_top = qdpll->assigned_vars + old_count;
  qdpll->bcp_ptr = qdpll->assigned_vars + old_bcp_index;
  qdpll->old_bcp_ptr = qdpll->assigned_vars + old_old_bcp_index;
}


static void
push_assigned_vars (QDPLL * qdpll, VarID id)
{
  if (qdpll->assigned_vars_top == qdpll->assigned_vars_end)
    enlarge_assigned_vars (qdpll);
  assert (qdpll->assigned_vars_top < qdpll->assigned_vars_end);
  assert (qdpll->assigned_vars <= qdpll->assigned_vars_top);

  Var *var = VARID2VARPTR (qdpll->pcnf.vars, id);
  assert (var->trail_pos == QDPLL_INVALID_TRAIL_POS);
  var->trail_pos = qdpll->assigned_vars_top - qdpll->assigned_vars;

  *(qdpll->assigned_vars_top++) = id;
}


/* -------------------- START: INEFFICIENT STATE CHECK -------------------- */

static int
is_clause_empty (QDPLL * qdpll, Constraint * clause)
{
  assert (!clause->is_cube);
  Var *vars = qdpll->pcnf.vars;

  LitID *p, *e;
  for (p = clause->lits, e = p + clause->num_lits; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);

      if (!QDPLL_VAR_ASSIGNED (var))
	{
	  if (QDPLL_VAR_EXISTS (var))
	    return 0;
	}
      else
	{
	  if (QDPLL_LIT_NEG (lit))
	    {
	      if (QDPLL_VAR_ASSIGNED_FALSE (var))
		return 0;
	    }
	  else
	    {
	      assert (QDPLL_LIT_POS (lit));
	      if (QDPLL_VAR_ASSIGNED_TRUE (var))
		return 0;
	    }
	}
    }

  return 1;
}

static void
remove_clause_from_notify_list (QDPLL * qdpll, const int is_cube,
				int lit_is_rwlit, LitID lit,
				Constraint * clause);

static void
add_clause_to_notify_list (QDPLL * qdpll, const int is_cube, int lit_is_rwlit,
			   LitID lit, Var * var, Constraint * clause);

static void
adjust_literal_watchers_in_sat_clause (QDPLL * qdpll,
				       Constraint * clause,
				       unsigned int sat_offset)
{
  assert (clause->lwatcher_pos != QDPLL_INVALID_WATCHER_POS);
  assert (clause->rwatcher_pos != QDPLL_INVALID_WATCHER_POS);
  assert (clause->lwatcher_pos < clause->rwatcher_pos);
  assert (clause->lwatcher_pos < clause->num_lits);
  assert (clause->rwatcher_pos < clause->num_lits);
  assert (sat_offset < clause->num_lits);
  const int is_cube = clause->is_cube;
  LitID sat_lit = clause->lits[sat_offset];
  Var *sat_var = LIT2VARPTR (qdpll->pcnf.vars, sat_lit);
  assert (is_cube
	  || (QDPLL_LIT_NEG (sat_lit) && QDPLL_VAR_ASSIGNED_FALSE (sat_var))
	  || (QDPLL_LIT_POS (sat_lit) && QDPLL_VAR_ASSIGNED_TRUE (sat_var)));
  assert (!is_cube
	  || (QDPLL_LIT_NEG (sat_lit) && QDPLL_VAR_ASSIGNED_TRUE (sat_var))
	  || (QDPLL_LIT_POS (sat_lit) && QDPLL_VAR_ASSIGNED_FALSE (sat_var)));

  if (clause->rwatcher_pos < sat_offset)
    {
      if ((!is_cube && QDPLL_SCOPE_FORALL (sat_var->scope)) ||
	  (is_cube && QDPLL_SCOPE_EXISTS (sat_var->scope)))
	return;
      /* Set right watcher. */
      LitID wlit = clause->lits[clause->rwatcher_pos];
      remove_clause_from_notify_list (qdpll, is_cube, 1, wlit, clause);
      add_clause_to_notify_list (qdpll, is_cube, 1, sat_lit, sat_var, clause);
      clause->rwatcher_pos = sat_offset;
    }
  else
    {
      /* Set left watcher. */
      LitID wlit = clause->lits[clause->lwatcher_pos];
      remove_clause_from_notify_list (qdpll, is_cube, 0, wlit, clause);
      add_clause_to_notify_list (qdpll, is_cube, 0, sat_lit, sat_var, clause);
      clause->lwatcher_pos = sat_offset;
    }

#ifndef NDEBUG
  LitID rwlit = clause->lits[clause->rwatcher_pos];
  Var *rwvar = LIT2VARPTR (qdpll->pcnf.vars, rwlit);
  assert (is_cube || QDPLL_SCOPE_EXISTS (rwvar->scope));
  assert (!is_cube || QDPLL_SCOPE_FORALL (rwvar->scope));
#endif
}


static int
is_clause_satisfied (QDPLL * qdpll, Constraint * clause)
{
  assert (clause);
  assert (!clause->is_cube);
  Var *vars = qdpll->pcnf.vars;
  int init_watchers = 0;

  if (clause->num_lits > 1
      && clause->lwatcher_pos != QDPLL_INVALID_WATCHER_POS
      && clause->rwatcher_pos != QDPLL_INVALID_WATCHER_POS)
    {
      init_watchers = 1;
      assert (clause->lwatcher_pos < clause->num_lits);
      assert (clause->rwatcher_pos < clause->num_lits);
      LitID wlit = *(clause->lits + clause->lwatcher_pos);
      Var *wvar = LIT2VARPTR (vars, wlit);
      if (QDPLL_LIT_NEG (wlit))
	{
	  if (QDPLL_VAR_ASSIGNED_FALSE (wvar))
	    {
	      return 1;
	    }
	}
      else
	{
	  assert (QDPLL_LIT_POS (wlit));
	  if (QDPLL_VAR_ASSIGNED_TRUE (wvar))
	    {
	      return 1;
	    }
	}
      wlit = *(clause->lits + clause->rwatcher_pos);
      wvar = LIT2VARPTR (vars, wlit);

      if (QDPLL_LIT_NEG (wlit))
	{
	  if (QDPLL_VAR_ASSIGNED_FALSE (wvar))
	    {
	      return 1;
	    }
	}
      else
	{
	  assert (QDPLL_LIT_POS (wlit));
	  if (QDPLL_VAR_ASSIGNED_TRUE (wvar))
	    {
	      return 1;
	    }
	}
    }

  LitID *p, *e;
  for (p = clause->lits, e = p + clause->num_lits; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);

      if (QDPLL_LIT_NEG (lit))
	{
	  if (QDPLL_VAR_ASSIGNED_FALSE (var))
	    {
	      if (init_watchers)
		adjust_literal_watchers_in_sat_clause (qdpll, clause,
						       p - clause->lits);
	      return 1;
	    }
	}
      else
	{
	  assert (QDPLL_LIT_POS (lit));
	  if (QDPLL_VAR_ASSIGNED_TRUE (var))
	    {
	      if (init_watchers)
		adjust_literal_watchers_in_sat_clause (qdpll, clause,
						       p - clause->lits);
	      return 1;
	    }
	}
    }

  return 0;
}


static int
is_clause_satisfied_by_prop_var (QDPLL * qdpll, Constraint * clause)
{
  assert (clause);
  assert (!clause->is_cube);
  Var *vars = qdpll->pcnf.vars;
  int init_watchers = 0;

  if (clause->num_lits > 1
      && clause->lwatcher_pos != QDPLL_INVALID_WATCHER_POS
      && clause->rwatcher_pos != QDPLL_INVALID_WATCHER_POS)
    {
      init_watchers = 1;
      assert (clause->lwatcher_pos < clause->num_lits);
      assert (clause->rwatcher_pos < clause->num_lits);
      LitID wlit = *(clause->lits + clause->lwatcher_pos);
      Var *wvar = LIT2VARPTR (vars, wlit);

      if (QDPLL_LIT_NEG (wlit))
	{
	  if (QDPLL_VAR_ASSIGNED_FALSE (wvar)
	      && QDPLL_VAR_MARKED_PROPAGATED (wvar))
	    {
	      return 1;
	    }
	}
      else
	{
	  assert (QDPLL_LIT_POS (wlit));
	  if (QDPLL_VAR_ASSIGNED_TRUE (wvar)
	      && QDPLL_VAR_MARKED_PROPAGATED (wvar))
	    {
	      return 1;
	    }
	}
      wlit = *(clause->lits + clause->rwatcher_pos);
      wvar = LIT2VARPTR (vars, wlit);

      if (QDPLL_LIT_NEG (wlit))
	{
	  if (QDPLL_VAR_ASSIGNED_FALSE (wvar)
	      && QDPLL_VAR_MARKED_PROPAGATED (wvar))
	    {
	      return 1;
	    }
	}
      else
	{
	  assert (QDPLL_LIT_POS (wlit));
	  if (QDPLL_VAR_ASSIGNED_TRUE (wvar)
	      && QDPLL_VAR_MARKED_PROPAGATED (wvar))
	    {
	      return 1;
	    }
	}
    }

  LitID *p, *e;
  for (p = clause->lits, e = p + clause->num_lits; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);

      if (QDPLL_LIT_NEG (lit))
	{
	  if (QDPLL_VAR_ASSIGNED_FALSE (var)
	      && QDPLL_VAR_MARKED_PROPAGATED (var))
	    {
	      if (init_watchers)
		adjust_literal_watchers_in_sat_clause (qdpll, clause,
						       p - clause->lits);
	      return 1;
	    }
	}
      else
	{
	  assert (QDPLL_LIT_POS (lit));
	  if (QDPLL_VAR_ASSIGNED_TRUE (var)
	      && QDPLL_VAR_MARKED_PROPAGATED (var))
	    {
	      if (init_watchers)
		adjust_literal_watchers_in_sat_clause (qdpll, clause,
						       p - clause->lits);
	      return 1;
	    }
	}
    }

  return 0;
}


/* Dual to 'is_clause_empty' */
static int
is_cube_satisfied (QDPLL * qdpll, Constraint * cube)
{
  assert (cube->is_cube);
  Var *vars = qdpll->pcnf.vars;

  LitID *p, *e;
  for (p = cube->lits, e = p + cube->num_lits; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      if (!QDPLL_VAR_ASSIGNED (var))
	{
	  if (QDPLL_VAR_FORALL (var))
	    return 0;
	}
      else
	{
	  if (QDPLL_LIT_NEG (lit))
	    {
	      if (QDPLL_VAR_ASSIGNED_TRUE (var))
		return 0;
	    }
	  else
	    {
	      assert (QDPLL_LIT_POS (lit));
	      if (QDPLL_VAR_ASSIGNED_FALSE (var))
		return 0;
	    }
	}
    }

  return 1;
}


/* Dual to 'is_clause_satisfied' */
static int
is_cube_empty (QDPLL * qdpll, Constraint * cube)
{
  assert (cube->is_cube);
  Var *vars = qdpll->pcnf.vars;
  assert (cube->lwatcher_pos < cube->num_lits);
  assert (cube->rwatcher_pos < cube->num_lits);
  LitID wlit = *(cube->lits + cube->lwatcher_pos);
  Var *wvar = LIT2VARPTR (vars, wlit);

  if (QDPLL_LIT_NEG (wlit))
    {
      if (QDPLL_VAR_ASSIGNED_TRUE (wvar))
	{
	  return 1;
	}
    }
  else
    {
      assert (QDPLL_LIT_POS (wlit));
      if (QDPLL_VAR_ASSIGNED_FALSE (wvar))
	{
	  return 1;
	}
    }
  wlit = *(cube->lits + cube->rwatcher_pos);
  wvar = LIT2VARPTR (vars, wlit);

  if (QDPLL_LIT_NEG (wlit))
    {
      if (QDPLL_VAR_ASSIGNED_TRUE (wvar))
	{
	  return 1;
	}
    }
  else
    {
      assert (QDPLL_LIT_POS (wlit));
      if (QDPLL_VAR_ASSIGNED_FALSE (wvar))
	{
	  return 1;
	}
    }

  LitID *p, *e;
  for (p = cube->lits, e = p + cube->num_lits; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);

      if (QDPLL_LIT_NEG (lit))
	{
	  if (QDPLL_VAR_ASSIGNED_TRUE (var))
	    {
	      adjust_literal_watchers_in_sat_clause (qdpll, cube,
						     p - cube->lits);
	      return 1;
	    }
	}
      else
	{
	  assert (QDPLL_LIT_POS (lit));
	  if (QDPLL_VAR_ASSIGNED_FALSE (var))
	    {
	      adjust_literal_watchers_in_sat_clause (qdpll, cube,
						     p - cube->lits);
	      return 1;
	    }
	}
    }

  return 0;
}

static int has_formula_empty_clause (QDPLL * qdpll);
static int has_constraint_spurious_pure_lit (QDPLL * qdpll, Constraint * c);

static int
has_formula_satisfied_cube (QDPLL * qdpll)
{
  Constraint *c;
  for (c = qdpll->pcnf.learnt_cubes.first; c; c = c->link.next)
    {
      if (is_cube_satisfied (qdpll, c))
	{
	  if (!has_constraint_spurious_pure_lit (qdpll, c))
	    return 1;
	}
    }
  return 0;
}


static int
has_formula_empty_clause (QDPLL * qdpll)
{
  Constraint *c;
  for (c = qdpll->pcnf.clauses.first; c; c = c->link.next)
    {
      if (is_clause_empty (qdpll, c))
	return 1;
    }
  for (c = qdpll->pcnf.learnt_clauses.first; c; c = c->link.next)
    {
      if (is_clause_empty (qdpll, c))
	{
	  if (!has_constraint_spurious_pure_lit (qdpll, c))
	    return 1;
	}
    }
  return 0;
}


static int
all_cubes_empty (QDPLL * qdpll)
{
  Constraint *c;
  for (c = qdpll->pcnf.learnt_cubes.first; c; c = c->link.next)
    {
      if (!is_cube_empty (qdpll, c))
	return 0;
    }
  return 1;
}


static int
all_clauses_satisfied (QDPLL * qdpll)
{
  Constraint *c;
  for (c = qdpll->pcnf.clauses.first; c; c = c->link.next)
    {
      if (!is_clause_satisfied (qdpll, c))
	return 0;
    }
  for (c = qdpll->pcnf.learnt_clauses.first; c; c = c->link.next)
    {
      if (!is_clause_satisfied (qdpll, c)
	  && (!has_constraint_spurious_pure_lit (qdpll, c)))
	return 0;
    }
  return 1;
}


static int
is_formula_false (QDPLL * qdpll)
{
  int has_empty_clause = has_formula_empty_clause (qdpll);
  if (qdpll->result_constraint && qdpll->result_constraint->is_cube)
    {
      assert (is_cube_satisfied (qdpll, qdpll->result_constraint));
      return 0;
    }
  if (has_empty_clause && (1 || all_cubes_empty (qdpll)))
    {
      return 1;
    }
  return 0;
}


static int
is_formula_true (QDPLL * qdpll)
{
  if (has_formula_satisfied_cube (qdpll))
    {
      return 1;
    }
  if (all_clauses_satisfied (qdpll))
    return 1;
  return 0;
}


static QDPLLSolverState
determine_solver_state (QDPLL * qdpll)
{
  if (is_formula_false (qdpll))
    return QDPLL_SOLVER_STATE_UNSAT;
  else if (is_formula_true (qdpll))
    return QDPLL_SOLVER_STATE_SAT;
  else
    return QDPLL_SOLVER_STATE_UNDEF;
}


/* -------------------- END: INEFFICIENT STATE CHECK -------------------- */


/* -------------------- START: CLAUSE WATCHING -------------------- */


/* Delete signed 'id' from notify-list in constant time. Variable
   'owner' owns the lists 'notify_list' and
   'offset_in_watched_clause'. */
static void
remove_id_from_notify_list (Var * vars, Var * owner, LitIDStack * notify_list,
			    VarIDStack * offset_in_watched_clause,
			    VarID del_pos, LitID signed_id)
{
  assert (signed_id != 0);
  assert (count_in_notify_clause_watcher_list (notify_list, signed_id) == 1);
  assert (QDPLL_COUNT_STACK (*notify_list) ==
	  QDPLL_COUNT_STACK (*offset_in_watched_clause));
  assert (del_pos < ((unsigned int) QDPLL_COUNT_STACK (*notify_list)));

  /* Delete notify-list entry by overwriting with last element.
     Must also copy entry in offset-in-watcher list. */

  LitID last = QDPLL_POP_STACK (*notify_list);
  VarID last_offset = QDPLL_POP_STACK (*offset_in_watched_clause);

  if (QDPLL_COUNT_STACK (*notify_list) == 0)
    {
      /* Stacks are empty now. */
      assert (del_pos == 0);
      assert (QDPLL_COUNT_STACK (*offset_in_watched_clause) == 0);
      assert (count_in_notify_clause_watcher_list (notify_list, signed_id) ==
	      0);
      return;
    }

  notify_list->start[del_pos] = last;
  offset_in_watched_clause->start[del_pos] = last_offset;

  /* Finally, since the offset of 'last' in the notify-list changed,
     we must also update its stored offset in 'offset_in_notify_list'. */
  Var *last_var = LIT2VARPTR (vars, last);
  VarIDStack *other_offset_in_notify_list = last < 0 ?
    &(last_var->neg_offset_in_notify_list) : &(last_var->
					       pos_offset_in_notify_list);

  other_offset_in_notify_list->start[last_offset] = del_pos;
  assert (count_in_notify_clause_watcher_list (notify_list, signed_id) == 0);
}


static int
is_constraint_empty_watcher (QDPLL * qdpll, Constraint * c)
{
  if ((!c->is_cube && is_clause_satisfied (qdpll, c)) ||
      (c->is_cube && is_cube_empty (qdpll, c)))
    return 1;
  return 0;
}


/* Remove signed 'id' from notify-lists of variables in old watched clause.
   Watched clause is watched by variable 'id' and is satisfied now.
*/
static void
remove_watching_var_from_notify_lists (QDPLL * qdpll, LitID signed_id,
				       Constraint * watched_clause)
{
  assert (watched_clause->is_watched);
  watched_clause->is_watched--;
  assert (watched_clause->is_cube
	  || !is_clause_empty (qdpll, watched_clause));
  assert (!watched_clause->is_cube
	  || !is_cube_satisfied (qdpll, watched_clause));
  assert (is_constraint_empty_watcher (qdpll, watched_clause));
  assert (signed_id != 0);
  Var *vars = qdpll->pcnf.vars;
  VarID id = signed_id < 0 ? -signed_id : signed_id;

  LitID *p, *e;
  VarIDStack *offset_in_notify_list = signed_id < 0 ?
    &(VARID2VARPTR (vars, id)->neg_offset_in_notify_list) :
    &(VARID2VARPTR (vars, id)->pos_offset_in_notify_list);
  assert (QDPLL_COUNT_STACK (*offset_in_notify_list) ==
	  watched_clause->num_lits);
  VarID *del_pos_ptr = offset_in_notify_list->start;
  for (p = watched_clause->lits, e = p + watched_clause->num_lits; p < e; p++)
    {
      LitID lit = *p;
      assert (lit != 0);
      assert (lit != -signed_id);
      Var *v = LIT2VARPTR (vars, lit);

      VarIDStack *offset_in_watched_clause;
      LitIDStack *notify_list;
      if (QDPLL_LIT_NEG (lit))
	{
	  if (!watched_clause->is_cube)
	    {
	      offset_in_watched_clause = &(v->neg_offset_in_watched_clause);
	      notify_list = &(v->neg_notify_clause_watchers);
	    }
	  else
	    {
	      offset_in_watched_clause = &(v->pos_offset_in_watched_clause);
	      notify_list = &(v->pos_notify_clause_watchers);
	    }
	}
      else
	{
	  assert (QDPLL_LIT_POS (lit));
	  if (!watched_clause->is_cube)
	    {
	      offset_in_watched_clause = &(v->pos_offset_in_watched_clause);
	      notify_list = &(v->pos_notify_clause_watchers);
	    }
	  else
	    {
	      offset_in_watched_clause = &(v->neg_offset_in_watched_clause);
	      notify_list = &(v->neg_notify_clause_watchers);
	    }
	}

      if (v->id != id)
	{
	  assert (notify_list->start[*del_pos_ptr] == signed_id);
	  remove_id_from_notify_list (vars, v, notify_list,
				      offset_in_watched_clause,
				      *del_pos_ptr, signed_id);
	}
      del_pos_ptr++;
    }
  /* Must clear the offset list, since this is needed for the new watcher then. */
  QDPLL_RESET_STACK (*offset_in_notify_list);
}


/* Add signed 'id' to notify-lists of variables in new watched clause.
   Sign of ID indicates which watcher to update for var 'id' after notification.
   Watched clause is now watched by variable 'id'.
*/
static void
add_watching_var_to_notify_lists (QDPLL * qdpll, LitID signed_id,
				  Constraint * watched_clause)
{
  watched_clause->is_watched++;
  assert (watched_clause->is_watched <= watched_clause->num_lits);
  assert (!is_constraint_empty_watcher (qdpll, watched_clause));
  assert (signed_id != 0);
  assert (constraint_has_lit (watched_clause, signed_id));
  Var *vars = qdpll->pcnf.vars;
  VarID id = QDPLL_LIT_NEG (signed_id) ? -signed_id : signed_id;
  QDPLLMemMan *mm = qdpll->mm;

  VarID offset = 0;
  VarIDStack *offset_in_notify_list = QDPLL_LIT_NEG (signed_id) ?
    &(VARID2VARPTR (vars, id)->neg_offset_in_notify_list) :
    &(VARID2VARPTR (vars, id)->pos_offset_in_notify_list);
  assert (QDPLL_COUNT_STACK (*offset_in_notify_list) == 0);

  LitID *p, *e;
  for (p = watched_clause->lits, e = p + watched_clause->num_lits; p < e; p++)
    {
      LitID lit = *p;
      assert (lit != 0);
      assert (lit != -signed_id);
      Var *v = LIT2VARPTR (vars, lit);

      if (v->id != id)
	{
	  VarIDStack *offset_in_watched_clause;
	  LitIDStack *notify_list;
	  if (QDPLL_LIT_NEG (lit))
	    {
	      if (!watched_clause->is_cube)
		{
		  offset_in_watched_clause =
		    &(v->neg_offset_in_watched_clause);
		  notify_list = &(v->neg_notify_clause_watchers);
		}
	      else
		{
		  offset_in_watched_clause =
		    &(v->pos_offset_in_watched_clause);
		  notify_list = &(v->pos_notify_clause_watchers);
		}
	    }
	  else
	    {
	      assert (QDPLL_LIT_POS (lit));
	      if (!watched_clause->is_cube)
		{
		  offset_in_watched_clause =
		    &(v->pos_offset_in_watched_clause);
		  notify_list = &(v->pos_notify_clause_watchers);
		}
	      else
		{
		  offset_in_watched_clause =
		    &(v->neg_offset_in_watched_clause);
		  notify_list = &(v->neg_notify_clause_watchers);
		}
	    }

	  assert (count_in_notify_clause_watcher_list
		  (notify_list, signed_id) == 0);
	  /* Store offsets. */
	  QDPLL_PUSH_STACK (mm, *offset_in_notify_list,
			    QDPLL_COUNT_STACK (*notify_list));
	  QDPLL_PUSH_STACK (mm, *offset_in_watched_clause, offset);
	  QDPLL_PUSH_STACK (mm, *notify_list, signed_id);
	  assert (count_in_notify_clause_watcher_list
		  (notify_list, signed_id) == 1);
	}
      else			/* Push dummy entry. */
	QDPLL_PUSH_STACK (mm, *offset_in_notify_list, 0);
      offset++;
    }
  assert (QDPLL_COUNT_STACK (*offset_in_notify_list) ==
	  watched_clause->num_lits);
}


static void
set_new_watcher (QDPLL * qdpll, LitID signed_id, OccList * occ_list,
		 Constraint * new_watcher, unsigned int offset)
{
  assert (signed_id != 0);
  /* Adapt assertions to watcher initialization. */
  assert (occ_list->first == new_watcher
	  || occ_list->first->is_cube
	  || !is_clause_empty (qdpll, occ_list->first));
  assert (occ_list->first == new_watcher || occ_list->first->is_cube
	  || is_clause_satisfied (qdpll, occ_list->first));
  assert (occ_list->first == new_watcher || !occ_list->first->is_cube
	  || !is_cube_satisfied (qdpll, occ_list->first));
  assert (occ_list->first == new_watcher || !occ_list->first->is_cube
	  || is_cube_empty (qdpll, occ_list->first));
  assert (!is_constraint_empty_watcher (qdpll, new_watcher));
  assert (offset < new_watcher->num_lits);
  assert (new_watcher->lits[offset] == signed_id);

  occ_unlink (signed_id, occ_list, new_watcher, offset);
  occ_link_first (signed_id, occ_list, new_watcher, offset);

  /* Add watching variable's ID to notify-lists of variable in watched
     clause. */
  add_watching_var_to_notify_lists (qdpll, signed_id, new_watcher);
}


static Constraint *
find_and_set_new_watcher (QDPLL * qdpll, LitID lit, OccList * occ_list,
			  OccList * old_watcher_list, const int init)
{
  OccListIterator it;
  OLITER_INIT (it, occ_list->first, occ_list->foffset, lit);
  assert (init || occ_list != old_watcher_list || OLITER_CUR (it));
  assert (init || occ_list != old_watcher_list
	  || is_constraint_empty_watcher (qdpll, OLITER_CUR (it)));
  /* Start search at second element. */
  if (!init && occ_list == old_watcher_list)
    OLITER_NEXT (it, lit);
  while (OLITER_CUR (it))
    {
      if (!is_constraint_empty_watcher (qdpll, OLITER_CUR (it)))
	{
	  /* We have found a new watcher. */
	  if (!init)
	    remove_watching_var_from_notify_lists (qdpll, lit,
						   old_watcher_list->first);
	  set_new_watcher (qdpll, lit, occ_list, OLITER_CUR (it), it.offset);
	  return OLITER_CUR (it);
	}
      OLITER_NEXT (it, lit);
    }

  return 0;
}


/* Notify clause-watching variables to find new watcher after assignment. */
static void
notify_clause_watching_variables (QDPLL * qdpll, LitIDStack * notify_list)
{
  Var *vars = qdpll->pcnf.vars, *v;
  LitID *p, *e;
  for (p = notify_list->start, e = notify_list->top; p < e; p++)
    {
      assert (*p != 0);
      LitID signed_id = *p;
      v = LIT2VARPTR (vars, signed_id);

      if (QDPLL_VAR_ASSIGNED (v))
	continue;

      OccList *occs, *next_occs;
      QDPLLAssignment pure_value;
      if (QDPLL_LIT_NEG (signed_id))
	{
	  pure_value = QDPLL_SCOPE_EXISTS (v->scope) ?
	    QDPLL_ASSIGNMENT_TRUE : QDPLL_ASSIGNMENT_FALSE;
	  /* Must find new neg-occ watcher. */
	  if (v->mark_is_neg_watching_cube)
	    {
	      /* First search neg-occ cubes, then neg-occ clauses. */
	      assert (is_cube_empty (qdpll, v->neg_occ_cubes.first));
	      occs = &(v->neg_occ_cubes);
	      next_occs = &(v->neg_occ_clauses);
	    }
	  else
	    {
	      /* First search neg-occ clauses, then neg-occ cubes. */
	      assert (is_clause_satisfied (qdpll, v->neg_occ_clauses.first));
	      occs = &(v->neg_occ_clauses);
	      next_occs = &(v->neg_occ_cubes);
	    }
	}
      else
	{
	  assert (QDPLL_LIT_POS (signed_id));
	  pure_value = QDPLL_SCOPE_EXISTS (v->scope) ?
	    QDPLL_ASSIGNMENT_FALSE : QDPLL_ASSIGNMENT_TRUE;
	  /* Must find new pos-occ watcher. */
	  if (v->mark_is_pos_watching_cube)
	    {
	      /* First search pos-occ cubes, then pos-occ clauses. */
	      assert (is_cube_empty (qdpll, v->pos_occ_cubes.first));
	      occs = &(v->pos_occ_cubes);
	      next_occs = &(v->pos_occ_clauses);
	    }
	  else
	    {
	      /* First search pos-occ clauses, then pos-occ cubes. */
	      assert (is_clause_satisfied (qdpll, v->pos_occ_clauses.first));
	      occs = &(v->pos_occ_clauses);
	      next_occs = &(v->pos_occ_cubes);
	    }
	}

#ifndef NDEBUG
      LitID *old_top = notify_list->top;
#endif
      Constraint *new_in_occs = 0, *new_in_next_occs = 0;
      if (!(new_in_occs =
	    find_and_set_new_watcher (qdpll, signed_id, occs, occs, 0)) &&
	  !(new_in_next_occs =
	    find_and_set_new_watcher (qdpll, signed_id, next_occs, occs, 0)))
	{
	  assert (!new_in_occs && !new_in_next_occs);
	  /* Variable has no active occurrences left -> is pure. */
	  push_assigned_variable (qdpll, v, pure_value, QDPLL_VARMODE_PURE);
	}
      else
	{
	  assert (new_in_occs || new_in_next_occs);
	  /* Invert flag to indicate that we found a new
	     watcher in the other occ-list. */
	  if (!new_in_occs)
	    {
	      assert (new_in_next_occs);
	      if (QDPLL_LIT_NEG (signed_id))
		v->mark_is_neg_watching_cube = !v->mark_is_neg_watching_cube;
	      else
		v->mark_is_pos_watching_cube = !v->mark_is_pos_watching_cube;
	    }
	  /* New watcher was set. */
#ifndef NDEBUG
	  assert (old_top == notify_list->top + 1);
#endif
	  /* Entry has been removed from list being traversed. */
	  e--;
	  p--;
	  /* Must check 'new' element which was copied there. */
	}
    }
}


/* Find clause watchers for each variable.
   This is only for initialization before solver starts. */
static void
init_clause_watchers (QDPLL * qdpll)
{
  Var *vars = qdpll->pcnf.vars;

  Scope *s;
  for (s = qdpll->pcnf.scopes.first; s; s = s->link.next)
    {
      VarID *p, *e;
      for (p = s->vars.start, e = s->vars.top; p < e; p++)
	{
	  assert (*p > 0 && *p < qdpll->pcnf.size_vars);
	  Var *v = VARID2VARPTR (vars, *p);

	  if (QDPLL_VAR_ASSIGNED (v))
	    continue;

	  Constraint *watcher;

	  if ((watcher =
	       find_and_set_new_watcher (qdpll, -v->id, &(v->neg_occ_clauses),
					 &(v->neg_occ_clauses), 1)))
	    {
	      ;
	    }
	  else
	    {
	      /* Pure literal detected: variable has no negative occurrences. */
	      assert (!QDPLL_VAR_ASSIGNED (v));
	      if (QDPLL_VAR_EXISTS (v))
		push_assigned_variable (qdpll, v, QDPLL_ASSIGNMENT_TRUE,
					QDPLL_VARMODE_PURE);
	      else
		push_assigned_variable (qdpll, v, QDPLL_ASSIGNMENT_FALSE,
					QDPLL_VARMODE_PURE);
	      /* 'continue' here because: other watcher can not be set
	         since now all clauses implicitly satisfied. And we must
	         not enqueue two assignments. */
	      continue;
	    }

	  if ((watcher =
	       find_and_set_new_watcher (qdpll, v->id, &(v->pos_occ_clauses),
					 &(v->pos_occ_clauses), 1)))
	    {
	      ;
	    }
	  else
	    {
	      /* Pure literal detected: variable has no positive occurrences. */
	      assert (!QDPLL_VAR_ASSIGNED (v));
	      if (QDPLL_VAR_EXISTS (v))
		push_assigned_variable (qdpll, v, QDPLL_ASSIGNMENT_FALSE,
					QDPLL_VARMODE_PURE);
	      else
		push_assigned_variable (qdpll, v, QDPLL_ASSIGNMENT_TRUE,
					QDPLL_VARMODE_PURE);
	    }
	}
    }
}

/* -------------------- END: CLAUSE WATCHING -------------------- */


/* -------------------- START: LITERAL WATCHING -------------------- */

/* Traverse clause's literals between 'right' and 'left' and search
   unassigned literal of specified type. If a true literal is found,
   then value 'QDPLL_WATCHER_SAT' is returned.  */
static unsigned int
find_watcher_pos (QDPLL * qdpll, const int is_cube, Var * vars, LitID * lits,
		  LitID * right, LitID * left,
		  const QDPLLQuantifierType qtype, int *sat)
{
  Var *oldw = 0;
  QDPLLQuantifierType oldw_type = QDPLL_QTYPE_UNDEF;
  if (qtype == QDPLL_QTYPE_UNDEF && !(*sat))
    {
      /* Only when searching new left watcher. */
      oldw = LIT2VARPTR (qdpll->pcnf.vars, *(right + 1));
      oldw_type = oldw->scope->type;
      assert (!is_cube || oldw_type == QDPLL_QTYPE_FORALL);
      assert (is_cube || oldw_type == QDPLL_QTYPE_EXISTS);
    }

  for (; right >= left; right--)
    {
      assert (right >= lits);
      LitID lit = *right;
      assert (lit != 0);
      Var *var = LIT2VARPTR (vars, lit);
      if (!QDPLL_VAR_ASSIGNED (var))
	{
	  /* Literal unassigned. */
	  if (qtype == QDPLL_QTYPE_UNDEF || qtype == var->scope->type)
	    {
	      if (qtype == QDPLL_QTYPE_UNDEF && oldw_type != var->scope->type
		  && !(*sat) && qdpll->dm->is_init (qdpll->dm))
		{
		  if (!qdpll->dm->depends (qdpll->dm, var->id, oldw->id))
		    {
		      continue;
		    }
		}
	      return right - lits;
	    }
	}
      else
	{
	  /* Check if assigned literal satisfies clause / falsifies cube. */
	  if (QDPLL_LIT_NEG (lit))
	    {
	      if ((!is_cube && QDPLL_VAR_ASSIGNED_FALSE (var)) ||
		  (is_cube && QDPLL_VAR_ASSIGNED_TRUE (var)))
		{
		  if (qtype == QDPLL_QTYPE_UNDEF || qtype == var->scope->type)
		    {
		      *sat = 1;
		      return right - lits;
		    }
		  else
		    return QDPLL_WATCHER_SAT;
		}
	    }
	  else
	    {
	      assert (QDPLL_LIT_POS (lit));
	      if ((!is_cube && QDPLL_VAR_ASSIGNED_TRUE (var)) ||
		  (is_cube && QDPLL_VAR_ASSIGNED_FALSE (var)))
		{
		  if (qtype == QDPLL_QTYPE_UNDEF || qtype == var->scope->type)
		    {
		      *sat = 1;
		      return right - lits;
		    }
		  else
		    return QDPLL_WATCHER_SAT;
		}
	    }
	}
    }

  return QDPLL_INVALID_WATCHER_POS;
}


/* Delete the 'clause' from the literal's notify-list. Parameter
   'lit_is_rwlit' indicates if 'lit' is the literal of the right or
   left watcher. This avoids retrieving the watcher list again from
   the clause. */
static void
remove_clause_from_notify_list (QDPLL * qdpll, const int is_cube,
				int lit_is_rwlit, LitID lit,
				Constraint * clause)
{
  Var *vars = qdpll->pcnf.vars;
  Var *var = LIT2VARPTR (vars, lit);
  ConstraintPtrStack *notify_list;
  if (QDPLL_LIT_NEG (lit))
    {
      if (!is_cube)
	notify_list = &(var->pos_notify_lit_watchers);
      else
	notify_list = &(var->neg_notify_lit_watchers);
    }
  else
    {
      assert (QDPLL_LIT_POS (lit));
      if (!is_cube)
	notify_list = &(var->neg_notify_lit_watchers);
      else
	notify_list = &(var->pos_notify_lit_watchers);
    }
  assert (count_in_notify_literal_watcher_list (notify_list, clause) == 1);

  unsigned int offset = clause->offset_in_notify_list[lit_is_rwlit];
  Constraint *last_entry = QDPLL_POP_STACK (*notify_list);

  if (last_entry == clause)
    return;

  const int same_types = (clause->is_cube == last_entry->is_cube);
  /* Overwrite the current position with the last entry. */
  assert (notify_list->start[offset] == clause);
  notify_list->start[offset] = last_entry;
  /* Must update position information in the copied entry. */
  unsigned int *other_offsetp = last_entry->offset_in_notify_list;
  LitID other_wlit = *(last_entry->lits + last_entry->lwatcher_pos);
  /* Literal 'lit' must be watched in 'last_entry' as well. */
  if ((same_types && other_wlit != lit)
      || ((!same_types && other_wlit != -lit)))
    {
      other_wlit = *(last_entry->lits + last_entry->rwatcher_pos);
      other_offsetp++;
    }
  assert (!same_types || other_wlit == lit);
  assert (same_types || other_wlit == -lit);
  *other_offsetp = offset;

  assert (count_in_notify_literal_watcher_list (notify_list, clause) == 0);
}


static void
add_clause_to_notify_list (QDPLL * qdpll, const int is_cube, int lit_is_rwlit,
			   LitID lit, Var * var, Constraint * clause)
{
  QDPLLMemMan *mm = qdpll->mm;
  /* Add clause to notification list wrt. sign of literal. */
  ConstraintPtrStack *other_notify_list;
  if (QDPLL_LIT_NEG (lit))
    {
      if (!is_cube)
	other_notify_list = &(var->pos_notify_lit_watchers);
      else
	other_notify_list = &(var->neg_notify_lit_watchers);
    }
  else
    {
      assert (QDPLL_LIT_POS (lit));
      if (!is_cube)
	other_notify_list = &(var->neg_notify_lit_watchers);
      else
	other_notify_list = &(var->pos_notify_lit_watchers);
    }
  assert (count_in_notify_literal_watcher_list (other_notify_list, clause) ==
	  0);
  /* Store clauses's position in notify-list. */
  clause->offset_in_notify_list[lit_is_rwlit] =
    QDPLL_COUNT_STACK (*other_notify_list);
  QDPLL_PUSH_STACK (mm, *other_notify_list, clause);
  assert (count_in_notify_literal_watcher_list (other_notify_list, clause) ==
	  1);
}


/* Function is called to check satisfied cubes/learnt clauses and
   learnt unit constraints for spurious pure literals. */
static int
has_constraint_spurious_pure_lit (QDPLL * qdpll, Constraint * c)
{
  /* Spurious pure literals can only occur in learnt constraints
     because we use original clauses for detection. */
  if (!c->learnt)
    {
      assert (!c->is_cube);
      return 0;
    }
  Var *vars = qdpll->pcnf.vars;
  const int is_cube = c->is_cube;
  LitID *p, *e;
  for (p = c->lits, e = p + c->num_lits; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      if (var->mode == QDPLL_VARMODE_PURE)
	{
	  assert (QDPLL_VAR_ASSIGNED (var));
	  if (!is_cube && QDPLL_SCOPE_EXISTS (var->scope))
	    {
	      /* A false existential pure literal in a learnt clause
	         is always spurious. Normally, pure existential literals
	         always satisfy clauses. */
	      if ((QDPLL_LIT_NEG (lit) && QDPLL_VAR_ASSIGNED_TRUE (var)) ||
		  (QDPLL_LIT_POS (lit) && QDPLL_VAR_ASSIGNED_FALSE (var)))
		return 1;
	    }
	  else if (is_cube && QDPLL_SCOPE_FORALL (var->scope))
	    {
	      /* A true universal pure literal in a learnt cube
	         is always spurious. Normally, pure universal literals
	         always falsify cubes. */
	      if ((QDPLL_LIT_NEG (lit) && QDPLL_VAR_ASSIGNED_FALSE (var)) ||
		  (QDPLL_LIT_POS (lit) && QDPLL_VAR_ASSIGNED_TRUE (var)))
		return 1;
	    }
	}
    }
  return 0;
}


static void learnt_constraint_mtf (QDPLL * qdpll, Constraint * c);

static Constraint *
handle_detected_unit_constraint (QDPLL * qdpll, LitID lit, Var * var,
				 Constraint * constraint)
{
  assert (!QDPLL_VAR_ASSIGNED (var));
  assert (!QDPLL_VAR_MARKED_PROPAGATED (var));

  if (has_constraint_spurious_pure_lit (qdpll, constraint))
    {
      return constraint;
    }

  assert (!var->antecedent);
  var->antecedent = constraint;
  assert (!constraint->is_reason);
  constraint->is_reason = 1;
  if (constraint->learnt)
    {
      learnt_constraint_mtf (qdpll, constraint);
      constraint->unit_cnt++;
    }

  /* Push unit. */
  push_assigned_variable (qdpll, var,
			  QDPLL_LIT_POS (lit) ?
			  QDPLL_ASSIGNMENT_TRUE :
			  QDPLL_ASSIGNMENT_FALSE, QDPLL_VARMODE_UNIT);

  return constraint;
}


static Constraint *
update_literal_watchers (QDPLL * qdpll, Var * propagated_var,
			 Constraint * clause)
{
  QDPLLMemMan *mm = qdpll->mm;
  Var *vars = qdpll->pcnf.vars;
  const int is_cube = clause->is_cube;
  assert (clause->num_lits > 1);

  assert (QDPLL_VAR_ASSIGNED (propagated_var));
  assert (QDPLL_VAR_MARKED_PROPAGATED (propagated_var));
  assert (clause->num_lits == 1
	  || clause->lwatcher_pos < clause->rwatcher_pos);
  assert (clause->rwatcher_pos < clause->num_lits);
  assert (clause->lwatcher_pos < clause->num_lits);

  unsigned int oldlwpos = clause->lwatcher_pos;
  unsigned int newlwpos, newrwpos;

  LitID *lits = clause->lits;
  LitID lwlit = *(lits + oldlwpos);
  Var *lwvar = LIT2VARPTR (vars, lwlit);

  if (QDPLL_LIT_NEG (lwlit))
    {
      if ((!is_cube && QDPLL_VAR_ASSIGNED_FALSE (lwvar)) ||
	  (is_cube && QDPLL_VAR_ASSIGNED_TRUE (lwvar)))
	{
	  return clause;
	}
    }
  else
    {
      assert (QDPLL_LIT_POS (lwlit));
      if ((!is_cube && QDPLL_VAR_ASSIGNED_TRUE (lwvar)) ||
	  (is_cube && QDPLL_VAR_ASSIGNED_FALSE (lwvar)))
	{
	  return clause;
	}
    }

  unsigned int oldrwpos = clause->rwatcher_pos;
  LitID rwlit = *(lits + oldrwpos);
  Var *rwvar = LIT2VARPTR (vars, rwlit);

  if (QDPLL_LIT_NEG (rwlit))
    {
      if ((!is_cube && QDPLL_VAR_ASSIGNED_FALSE (rwvar)) ||
	  (is_cube && QDPLL_VAR_ASSIGNED_TRUE (rwvar)))
	{
	  return clause;
	}
    }
  else
    {
      assert (QDPLL_LIT_POS (rwlit));
      if ((!is_cube && QDPLL_VAR_ASSIGNED_TRUE (rwvar)) ||
	  (is_cube && QDPLL_VAR_ASSIGNED_FALSE (rwvar)))
	{
	  return clause;
	}
    }

  assert (is_cube || QDPLL_VAR_EXISTS (rwvar));
  assert (!is_cube || QDPLL_VAR_FORALL (rwvar));
  assert (QDPLL_VAR_ASSIGNED (lwvar) || QDPLL_VAR_ASSIGNED (rwvar));
  assert (rwlit != 0);
  assert (lwlit != 0);
  assert (clause->num_lits == 1 || rwlit != lwlit);
  assert (clause->num_lits == 1 || -rwlit != lwlit);

  int clause_sat = 0;

  if (!QDPLL_VAR_ASSIGNED (rwvar))
    {
      /* Left watcher assigned. Here, conflicts/solutions can NOT occur. */
      assert (lwvar == propagated_var);
      assert (is_cube || QDPLL_LIT_POS (lwlit)
	      || QDPLL_VAR_ASSIGNED_TRUE (lwvar));
      assert (is_cube || QDPLL_LIT_NEG (lwlit)
	      || QDPLL_VAR_ASSIGNED_FALSE (lwvar));
      assert (!is_cube || QDPLL_LIT_POS (lwlit)
	      || QDPLL_VAR_ASSIGNED_FALSE (lwvar));
      assert (!is_cube || QDPLL_LIT_NEG (lwlit)
	      || QDPLL_VAR_ASSIGNED_TRUE (lwvar));

      if ((newlwpos =
	   find_watcher_pos (qdpll, is_cube, vars, lits, lits + oldrwpos - 1,
			     lits, QDPLL_QTYPE_UNDEF,
			     &clause_sat)) != QDPLL_INVALID_WATCHER_POS)
	{
	  if (newlwpos != QDPLL_WATCHER_SAT)
	    {			/* New watcher found -> update notify lists. */
	      remove_clause_from_notify_list (qdpll, is_cube, 0, lwlit,
					      clause);
	      lwlit = *(lits + newlwpos);
	      assert (lwlit);
	      lwvar = LIT2VARPTR (vars, lwlit);
	      add_clause_to_notify_list (qdpll, is_cube, 0, lwlit, lwvar,
					 clause);
	      clause->lwatcher_pos = newlwpos;
	      return clause + 1;
	    }
	  else			/* Clause is satisfied. */
	    {
	      assert (0);
	      return clause;
	    }
	}
      else
	{
	  /* Did not find new left watcher. Next, try to find new
	     right watcher. */
	  newrwpos =
	    find_watcher_pos (qdpll, is_cube, vars, lits,
			      lits + clause->num_lits - 1, lits,
			      is_cube ? QDPLL_QTYPE_FORALL :
			      QDPLL_QTYPE_EXISTS, &clause_sat);
	  assert (newrwpos != QDPLL_INVALID_WATCHER_POS);
	  assert (oldrwpos <= newrwpos);
	  if (newrwpos != QDPLL_WATCHER_SAT)
	    {
	      if (newrwpos != oldrwpos)
		{
		  assert (oldrwpos < newrwpos);
		  newlwpos =
		    find_watcher_pos (qdpll, is_cube, vars, lits,
				      lits + newrwpos - 1, lits + oldrwpos,
				      QDPLL_QTYPE_UNDEF, &clause_sat);
		  assert (newlwpos != QDPLL_INVALID_WATCHER_POS);
		  assert (oldrwpos <= newlwpos);
		  if (newlwpos != QDPLL_WATCHER_SAT)
		    {
		      if (newlwpos != oldrwpos)
			{
			  /* Must remove entry for old watcher and add
			     two new entries. */
			  assert (oldrwpos < newlwpos);

			  remove_clause_from_notify_list (qdpll, is_cube, 1,
							  rwlit, clause);

			  rwlit = *(lits + newrwpos);
			  assert (rwlit);
			  rwvar = LIT2VARPTR (vars, rwlit);
			  add_clause_to_notify_list (qdpll, is_cube, 1, rwlit,
						     rwvar, clause);

			  remove_clause_from_notify_list (qdpll, is_cube, 0,
							  lwlit, clause);
			  lwlit = *(lits + newlwpos);
			  assert (lwlit);
			  lwvar = LIT2VARPTR (vars, lwlit);
			  add_clause_to_notify_list (qdpll, is_cube, 0, lwlit,
						     lwvar, clause);
			}
		      else
			{
			  /* Must add entry for new right watcher. */
			  remove_clause_from_notify_list (qdpll, is_cube, 0,
							  lwlit, clause);
			  clause->offset_in_notify_list[0] =
			    clause->offset_in_notify_list[1];
			  rwlit = *(lits + newrwpos);
			  assert (rwlit);
			  rwvar = LIT2VARPTR (vars, rwlit);
			  add_clause_to_notify_list (qdpll, is_cube, 1, rwlit,
						     rwvar, clause);
			}

		      clause->lwatcher_pos = newlwpos;
		      clause->rwatcher_pos = newrwpos;

		      return clause + 1;
		    }
		  else
		    {
		      /* Clause is satisfied. */
		      assert (0);
		      return clause;
		    }
		}
	      else
		{
		  /* Clause is unit. No unassigned literal to the left
		     of old rwatcher. */
		  assert (!clause_sat);
		  return handle_detected_unit_constraint (qdpll,
							  !is_cube ? rwlit :
							  -rwlit, rwvar,
							  clause);
		}
	    }
	  else
	    {
	      /* Clause is satisfied. */
	      return clause;
	    }
	}
    }
  else
    {
      /* Right watcher assigned. Here, both unit literals and
         conflicts can occur. */
      assert (is_cube || QDPLL_LIT_POS (rwlit)
	      || QDPLL_VAR_ASSIGNED_TRUE (rwvar));
      assert (is_cube || QDPLL_LIT_NEG (rwlit)
	      || QDPLL_VAR_ASSIGNED_FALSE (rwvar));
      assert (!is_cube || QDPLL_LIT_POS (rwlit)
	      || QDPLL_VAR_ASSIGNED_FALSE (rwvar));
      assert (!is_cube || QDPLL_LIT_NEG (rwlit)
	      || QDPLL_VAR_ASSIGNED_TRUE (rwvar));
      assert (QDPLL_VAR_ASSIGNED (lwvar) || rwvar == propagated_var);

      if ((newrwpos =
	   find_watcher_pos (qdpll, is_cube, vars, lits,
			     lits + clause->num_lits - 1, lits,
			     is_cube ? QDPLL_QTYPE_FORALL :
			     QDPLL_QTYPE_EXISTS,
			     &clause_sat)) != QDPLL_INVALID_WATCHER_POS)
	{
	  /* Clause can not be conflicting, since
	     existential literal found. */
	  if (newrwpos != QDPLL_WATCHER_SAT)
	    {
	      if ((newlwpos =
		   find_watcher_pos (qdpll, is_cube, vars, lits,
				     lits + newrwpos - 1, lits,
				     QDPLL_QTYPE_UNDEF,
				     &clause_sat)) !=
		  QDPLL_INVALID_WATCHER_POS)
		{
		  if (newlwpos != QDPLL_WATCHER_SAT)
		    {
		      /* New watcher found -> update notify lists. */
		      assert (newlwpos < newrwpos);

		      if (newlwpos != oldlwpos)
			{
			  /* Must remove one old entry and add two new ones. */
			  remove_clause_from_notify_list (qdpll, is_cube, 0,
							  lwlit, clause);
			  remove_clause_from_notify_list (qdpll, is_cube, 1,
							  rwlit, clause);

			  rwlit = *(lits + newrwpos);
			  assert (rwlit);
			  rwvar = LIT2VARPTR (vars, rwlit);
			  add_clause_to_notify_list (qdpll, is_cube, 1, rwlit,
						     rwvar, clause);

			  lwlit = *(lits + newlwpos);
			  assert (lwlit);
			  lwvar = LIT2VARPTR (vars, lwlit);
			  add_clause_to_notify_list (qdpll, is_cube, 0, lwlit,
						     lwvar, clause);
			}
		      else
			{	/* Add new entry for new right watcher. */
			  assert (clause->lwatcher_pos == newlwpos);
			  remove_clause_from_notify_list (qdpll, is_cube, 1,
							  rwlit, clause);
			  rwlit = *(lits + newrwpos);
			  assert (rwlit);
			  rwvar = LIT2VARPTR (vars, rwlit);
			  add_clause_to_notify_list (qdpll, is_cube, 1, rwlit,
						     rwvar, clause);
			}

		      clause->rwatcher_pos = newrwpos;
		      clause->lwatcher_pos = newlwpos;

		      return clause + 1;
		    }
		  else		/* Clause is satisfied. */
		    {
		      return clause;
		    }
		}
	      else
		{
		  if (!clause_sat)
		    {
		      rwlit = *(lits + newrwpos);
		      assert (rwlit != 0);
		      rwvar = LIT2VARPTR (vars, rwlit);

		      return handle_detected_unit_constraint (qdpll,
							      !is_cube ? rwlit
							      : -rwlit, rwvar,
							      clause);
		    }
		  else
		    {
		      if (oldlwpos < newrwpos)
			{
			  remove_clause_from_notify_list (qdpll, is_cube, 1,
							  rwlit, clause);
			  rwlit = *(lits + newrwpos);
			  assert (rwlit != 0);
			  rwvar = LIT2VARPTR (vars, rwlit);

			  add_clause_to_notify_list (qdpll, is_cube, 1, rwlit,
						     rwvar, clause);
			  clause->rwatcher_pos = newrwpos;
			  if (lwvar != propagated_var)
			    return clause + 1;
			  else
			    return clause;
			}
		      else
			{
			  return clause;
			}
		    }
		}
	    }
	  else
	    {
	      /* Clause is satisfied. */
	      return clause;
	    }
	}
      else
	{
	  /* Clause is conflicting: no free existential literal found. */
	  return 0;
	}
    }
}


static void
init_literal_watcher (QDPLL * qdpll, Constraint * c, unsigned int left_offset,
		      unsigned int right_offset)
{
  assert (c->num_lits > 0);
  assert (left_offset < c->num_lits);
  assert (right_offset < c->num_lits);
  assert (left_offset <= right_offset);
  assert (c->num_lits != 1 || left_offset == right_offset);
  assert (c->num_lits == 1 || left_offset != right_offset);

  QDPLLMemMan *mm = qdpll->mm;
  Var *vars = qdpll->pcnf.vars;
  LitID lit, *litp;
  VarID var_id;
  unsigned int num_lits = c->num_lits;
  const int is_cube = c->is_cube;
  Var *var;

  /* Set right watcher. */
  litp = c->lits + right_offset;
  assert (litp >= c->lits);
  assert (litp < c->lits + num_lits);
  lit = *litp;
  assert (lit != 0);
  var_id = LIT2VARID (lit);
  var = VARID2VARPTR (vars, var_id);
  assert (is_cube || QDPLL_VAR_EXISTS (var));
  assert (!is_cube || QDPLL_VAR_FORALL (var));

  c->rwatcher_pos = right_offset;

  /* Add clause to notification list wrt. sign of literal. */
  ConstraintPtrStack *notify_list;
  if (QDPLL_LIT_NEG (lit))
    {
      if (!is_cube)
	notify_list = &(var->pos_notify_lit_watchers);
      else
	notify_list = &(var->neg_notify_lit_watchers);
    }
  else
    {
      assert (QDPLL_LIT_POS (lit));
      if (!is_cube)
	notify_list = &(var->neg_notify_lit_watchers);
      else
	notify_list = &(var->pos_notify_lit_watchers);
    }
  assert (count_in_notify_literal_watcher_list (notify_list, c) == 0);
  /* Store clauses's position in notify-list. */
  c->offset_in_notify_list[1] = QDPLL_COUNT_STACK (*notify_list);
  QDPLL_PUSH_STACK (mm, *notify_list, c);
  assert (count_in_notify_literal_watcher_list (notify_list, c) == 1);

  /* Set left watcher. */
  litp = c->lits + left_offset;

  assert (litp >= c->lits);
  assert (litp < c->lits + num_lits);
  lit = *litp;
  assert (lit != 0);
  var_id = LIT2VARID (lit);
  var = VARID2VARPTR (vars, var_id);

  c->lwatcher_pos = left_offset;

  /* Add clause to notification list wrt. sign of literal. */
  if (QDPLL_LIT_NEG (lit))
    {
      if (!is_cube)
	notify_list = &(var->pos_notify_lit_watchers);
      else
	notify_list = &(var->neg_notify_lit_watchers);
    }
  else
    {
      assert (QDPLL_LIT_POS (lit));
      if (!is_cube)
	notify_list = &(var->neg_notify_lit_watchers);
      else
	notify_list = &(var->pos_notify_lit_watchers);
    }
  assert (num_lits == 1
	  || count_in_notify_literal_watcher_list (notify_list, c) == 0);
  /* Store clauses's position in notify-list. */
  c->offset_in_notify_list[0] = QDPLL_COUNT_STACK (*notify_list);
  QDPLL_PUSH_STACK (mm, *notify_list, c);
  assert (num_lits == 1
	  || count_in_notify_literal_watcher_list (notify_list, c) == 1);
}


static QDPLLSolverState
init_literal_watchers_for_constraint (QDPLL * qdpll, Constraint * c)
{
  assert (c->num_lits >= 1);
  Var *vars = qdpll->pcnf.vars;
  LitID lit, *litp;
  VarID var_id;
  unsigned int num_lits = c->num_lits;
  const int is_cube = c->is_cube;
  Var *var;
  if (num_lits == 1)
    {
      /* Collect existential unit literals. */
      lit = c->lits[0];
      assert (lit != 0);
      assert ((unsigned int) (LIT2VARID (lit)) < qdpll->pcnf.size_vars);
      var = LIT2VARPTR (vars, lit);
      assert (c->is_cube || QDPLL_VAR_EXISTS (var));
      assert (qdpll->state.decision_level == 0);

      if (QDPLL_LIT_NEG (lit))
	{
	  if (QDPLL_VAR_ASSIGNED_TRUE (var))
	    {
	      if (!is_cube)
		return QDPLL_SOLVER_STATE_UNSAT;
	    }
	  else if (QDPLL_VAR_ASSIGNED_FALSE (var))
	    {
	      if (!is_cube)
		return QDPLL_SOLVER_STATE_UNDEF;
	      else
		return QDPLL_SOLVER_STATE_SAT;
	    }
	  else
	    {
	      assert (!var->antecedent);
	      var->antecedent = c;
	      assert (!c->is_reason);
	      c->is_reason = 1;
	      push_assigned_variable (qdpll, var,
				      is_cube ? QDPLL_ASSIGNMENT_TRUE :
				      QDPLL_ASSIGNMENT_FALSE,
				      QDPLL_VARMODE_UNIT);
	    }
	}
      else
	{
	  if (QDPLL_VAR_ASSIGNED_FALSE (var))
	    {
	      if (!is_cube)
		return QDPLL_SOLVER_STATE_UNSAT;
	    }
	  else if (QDPLL_VAR_ASSIGNED_TRUE (var))
	    {
	      if (!is_cube)
		return QDPLL_SOLVER_STATE_UNDEF;
	      else
		return QDPLL_SOLVER_STATE_SAT;
	    }
	  else
	    {
	      assert (!var->antecedent);
	      var->antecedent = c;
	      assert (!c->is_reason);
	      c->is_reason = 1;
	      push_assigned_variable (qdpll, var,
				      is_cube ? QDPLL_ASSIGNMENT_FALSE :
				      QDPLL_ASSIGNMENT_TRUE,
				      QDPLL_VARMODE_UNIT);
	    }
	}
      assert (qdpll->state.decision_level == 0);
    }
  else
    {
      unsigned int right_offset = c->num_lits - 1;
      unsigned int left_offset = right_offset;
      if (num_lits > 1)
	left_offset--;
      init_literal_watcher (qdpll, c, left_offset, right_offset);
    }

  return QDPLL_SOLVER_STATE_UNDEF;
}



static QDPLLSolverState
init_literal_watchers (QDPLL * qdpll)
{
  QDPLLMemMan *mm = qdpll->mm;
  Var *vars = qdpll->pcnf.vars;
  QDPLLSolverState result;
  Constraint *c;
  for (c = qdpll->pcnf.clauses.first; c; c = c->link.next)
    {
      if ((result =
	   init_literal_watchers_for_constraint (qdpll,
						 c)) !=
	  QDPLL_SOLVER_STATE_UNDEF)
	return result;
    }

  for (c = qdpll->pcnf.learnt_clauses.first; c; c = c->link.next)
    {
      if ((result =
	   init_literal_watchers_for_constraint (qdpll,
						 c)) !=
	  QDPLL_SOLVER_STATE_UNDEF)
	return result;
    }

  for (c = qdpll->pcnf.learnt_cubes.first; c; c = c->link.next)
    {
      if ((result =
	   init_literal_watchers_for_constraint (qdpll,
						 c)) !=
	  QDPLL_SOLVER_STATE_UNDEF)
	return result;
    }

  return QDPLL_SOLVER_STATE_UNDEF;
}

/* -------------------- END: LITERAL WATCHING -------------------- */


static void
delete_scope (QDPLL * qdpll, Scope * scope)
{
  QDPLLMemMan *mm = qdpll->mm;
  QDPLL_DELETE_STACK (mm, scope->vars);
  qdpll_free (mm, scope, sizeof (Scope));
}


/* Remove scopes which contain no variable. 
   Typically called after no-occ variables have been eliminated.
   Runtime is worst-case quadratic in the number of scopes.
*/
static void
cleanup_empty_scopes (QDPLL * qdpll)
{
  Scope *s, *n;
  for (s = qdpll->pcnf.scopes.first; s; s = n)
    {
      n = s->link.next;

      assert (s->nesting != QDPLL_DEFAULT_SCOPE_NESTING
	      || QDPLL_SCOPE_EXISTS (s));

      /* Should keep one outermost existential scope as default scope. */
      if (!QDPLL_COUNT_STACK (s->vars)
	  && s->nesting != QDPLL_DEFAULT_SCOPE_NESTING)
	{			/* Unlink and delete scope. */
	  UNLINK (qdpll->pcnf.scopes, s, link);
	  delete_scope (qdpll, s);
	  /* Save next scope 'n' for continuing in next loop iteration. */
	  s = n;
	  for (; n; n = n->link.next)
	    n->nesting--;
	  n = s;
	}
    }
}


static void
delete_variable (QDPLL * qdpll, Var * var)
{
  QDPLLMemMan *mm = qdpll->mm;
  QDPLL_DELETE_STACK (mm, var->pos_notify_clause_watchers);
  QDPLL_DELETE_STACK (mm, var->neg_notify_clause_watchers);
  QDPLL_DELETE_STACK (mm, var->pos_offset_in_notify_list);
  QDPLL_DELETE_STACK (mm, var->neg_offset_in_notify_list);
  QDPLL_DELETE_STACK (mm, var->pos_offset_in_watched_clause);
  QDPLL_DELETE_STACK (mm, var->neg_offset_in_watched_clause);
  QDPLL_DELETE_STACK (mm, var->pos_notify_lit_watchers);
  QDPLL_DELETE_STACK (mm, var->neg_notify_lit_watchers);

  QDPLL_DELETE_STACK (mm, var->type_red_members);

  QDPLLDepManGeneric *dm = qdpll->dm;
  assert (dm);
  dm->notify_reset_variable (dm, var->id);
}


static void
reset_variable (QDPLL * qdpll, Var * var)
{
  delete_variable (qdpll, var);
  assert (qdpll->pcnf.used_vars != 0);
  qdpll->pcnf.used_vars--;
  memset (var, 0, sizeof (Var));
}


/* Remove variables without occurrences. */
static void
cleanup_no_occ_variables (QDPLL * qdpll)
{
  Var *vars = qdpll->pcnf.vars;

  Scope *s;
  for (s = qdpll->pcnf.scopes.first; s; s = s->link.next)
    {
      VarIDStack *scope_vars = &s->vars;
      VarID *p, *end, *last;
      for (p = scope_vars->start, end = scope_vars->top, last = end - 1;
	   p < end; p++)
	{
	  assert (*p > 0);
	  Var *v = VARID2VARPTR (vars, *p);
	  if (!QDPLL_VAR_HAS_OCCS (v))
	    {
	      /* Variable must not be on priority queue. */
	      assert (v->priority_pos == QDPLL_INVALID_PQUEUE_POS);
	      reset_variable (qdpll, v);
	      *p-- = *last--;
	      end--;
	      scope_vars->top--;
	    }
	}
    }
}


/* Maintain prefix properties.
   Should be called before solving starts.
   This matters mostly for the dependency manager, not for the solver itself.
   Runtime is worst-case quadratic in the number of scopes.
*/
static void
merge_adjacent_same_type_scopes (QDPLL * qdpll)
{
  QDPLLMemMan *mm = qdpll->mm;

  Scope *s, *n;
  for (s = qdpll->pcnf.scopes.first; s; s = n)
    {
      n = s->link.next;
      assert (s->nesting == QDPLL_DEFAULT_SCOPE_NESTING
	      || QDPLL_COUNT_STACK (s->vars));
      assert (!n || QDPLL_COUNT_STACK (n->vars));

      if (n && s->type == n->type)
	{
	  /* Adjacent scopes have same type -> merge. */
	  VarIDStack *scope_vars = &s->vars;
	  VarID *p, *e, v;
	  for (p = n->vars.start, e = n->vars.top; p < e; p++)
	    {
	      v = *p;
	      QDPLL_PUSH_STACK (mm, *scope_vars, v);
	      assert (qdpll->pcnf.vars[v].scope == n);
	      qdpll->pcnf.vars[v].scope = s;
	    }

	  UNLINK (qdpll->pcnf.scopes, n, link);
	  delete_scope (qdpll, n);

	  for (n = s->link.next; n; n = n->link.next)
	    n->nesting--;
	  n = s;
	}
    }
}


/* Cleanup formula. */
static void
clean_up_formula (QDPLL * qdpll)
{
  cleanup_no_occ_variables (qdpll);
  cleanup_empty_scopes (qdpll);
  merge_adjacent_same_type_scopes (qdpll);
}


static void
reset_watchers (QDPLL * qdpll)
{
  Constraint *c;
  for (c = qdpll->pcnf.clauses.first; c; c = c->link.next)
    {
      c->is_watched = 0;
      c->rwatcher_pos = c->lwatcher_pos = QDPLL_INVALID_WATCHER_POS;
      c->offset_in_notify_list[0] = c->offset_in_notify_list[1] = 0;
    }

  for (c = qdpll->pcnf.learnt_clauses.first; c; c = c->link.next)
    {
      c->is_watched = 0;
      c->rwatcher_pos = c->lwatcher_pos = QDPLL_INVALID_WATCHER_POS;
      c->offset_in_notify_list[0] = c->offset_in_notify_list[1] = 0;
    }

  for (c = qdpll->pcnf.learnt_cubes.first; c; c = c->link.next)
    {
      c->is_watched = 0;
      c->rwatcher_pos = c->lwatcher_pos = QDPLL_INVALID_WATCHER_POS;
      c->offset_in_notify_list[0] = c->offset_in_notify_list[1] = 0;
    }

  Var *p, *e;
  for (p = qdpll->pcnf.vars, e = p + qdpll->pcnf.size_vars; p < e; p++)
    {
      if (p->id)
	{
	  p->mark_is_neg_watching_cube = p->mark_is_pos_watching_cube = 0;
	  QDPLL_RESET_STACK (p->pos_notify_clause_watchers);
	  QDPLL_RESET_STACK (p->neg_notify_clause_watchers);
	  QDPLL_RESET_STACK (p->pos_offset_in_notify_list);
	  QDPLL_RESET_STACK (p->neg_offset_in_notify_list);
	  QDPLL_RESET_STACK (p->pos_offset_in_watched_clause);
	  QDPLL_RESET_STACK (p->neg_offset_in_watched_clause);
	  QDPLL_RESET_STACK (p->pos_notify_lit_watchers);
	  QDPLL_RESET_STACK (p->neg_notify_lit_watchers);
	}
    }
}


static QDPLLSolverState
set_up_watchers (QDPLL * qdpll)
{
  init_clause_watchers (qdpll);
  QDPLLSolverState state = init_literal_watchers (qdpll);
  return state;
}


/* Clean up formula and do initialization tasks:
   Remove no-occ variables and empty scopes, 
   merge scopes of same type into one scope.
*/
static QDPLLSolverState
set_up_formula (QDPLL * qdpll)
{
  clean_up_formula (qdpll);
  return set_up_watchers (qdpll);
}


/* Set variable ID and scope and add to scope. */
static void
declare_and_init_variable (QDPLL * qdpll, Scope * scope, VarID id)
{
  assert (id > 0);
  assert (id < qdpll->pcnf.size_vars);

  QDPLLMemMan *mm = qdpll->mm;
  Var *var = VARID2VARPTR (qdpll->pcnf.vars, id);

  qdpll->pcnf.used_vars++;

  /* Init variable */
  assert (!var->id);
  var->id = id;
  assert (!var->scope);
  var->scope = scope;
  assert (!var->priority_pos);
  var->priority_pos = QDPLL_INVALID_PQUEUE_POS;
  assert (!var->priority);
  var->priority = 1;
  assert (!var->decision_level);
  var->decision_level = QDPLL_INVALID_DECISION_LEVEL;
  assert (!var->trail_pos);
  var->trail_pos = QDPLL_INVALID_TRAIL_POS;

  assert (!QDPLL_VAR_HAS_POS_OCCS (var));
  assert (!QDPLL_VAR_HAS_NEG_OCCS (var));
  assert (!QDPLL_VAR_HAS_POS_OCC_CUBES (var));
  assert (!QDPLL_VAR_HAS_NEG_OCC_CUBES (var));

  /* Add to scope */
  QDPLL_PUSH_STACK (mm, scope->vars, id);

  /* Inform DepMan that new variable has been declared and initialized. */
  QDPLLDepManGeneric *dm = qdpll->dm;
  assert (dm);
  dm->notify_init_variable (dm, id);

  if (id > qdpll->pcnf.max_declared_var_id)
    qdpll->pcnf.max_declared_var_id = id;
}


static int
compare_lits_by_variable_nesting (QDPLL * qdpll, LitID lit1, LitID lit2)
{
  Var *vars = qdpll->pcnf.vars;
  VarID var_id1 = LIT2VARID (lit1);
  VarID var_id2 = LIT2VARID (lit2);
  Var *var1 = VARID2VARPTR (vars, var_id1);
  Var *var2 = VARID2VARPTR (vars, var_id2);

  unsigned int nesting1 = var1->scope->nesting;
  unsigned int nesting2 = var2->scope->nesting;

  if (nesting1 < nesting2)
    return -1;
  else if (nesting1 > nesting2)
    return 1;
  else
    {
      if (var_id1 < var_id2)
	return -1;
      else if (var_id1 > var_id2)
	return 1;
      else
	return 0;
    }
}


static void
unmark_clause_variables (QDPLL * qdpll, Constraint * clause)
{
  assert (!clause->is_cube);
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *end, lit;
  for (p = clause->lits, end = p + clause->num_lits; p < end; p++)
    {				/* Unmark variables */
      lit = *p;
      QDPLL_VAR_UNMARK (LIT2VARPTR (vars, lit));
    }
}


/* Apply universal reduction. */
static Constraint *
universally_reduce_clause (QDPLL * qdpll, Constraint * clause)
{
  assert (!clause->is_cube);
  Var *vars = qdpll->pcnf.vars;
  LitID lit, *p, *e;
  for (e = clause->lits, p = e + clause->num_lits - 1; p >= e; p--)
    {
      lit = *p;
      Var *v = LIT2VARPTR (vars, lit);
      if (QDPLL_SCOPE_FORALL (v->scope))
	clause->num_lits--;
      else
	break;
    }

  assert (clause->num_lits == 0 ||
	  QDPLL_SCOPE_EXISTS (LIT2VARPTR (vars,
					  clause->lits[clause->num_lits -
						       1])->scope));

  if (clause->num_lits == 0)
    return clause;
  else
    return 0;
}


/* Check clause for multiple, complementary literals and universal-reduction.
   Returns 'NULL' if clause is not tautological, 
   otherwise returns pointer to tautological clause. 
*/
static Constraint *
cleanup_clause (QDPLL * qdpll, Constraint * clause)
{
  assert (!clause->is_cube);
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *end, *last, lit;
  for (p = clause->lits, end = p + clause->num_lits, last = end - 1;
       p < end; p++)
    {
      lit = *p;
      assert (lit != 0);
      Var *var = LIT2VARPTR (vars, lit);
      if (!QDPLL_VAR_MARKED (var))
	{
	  if (lit < 0)
	    QDPLL_VAR_NEG_MARK (var);
	  else
	    QDPLL_VAR_POS_MARK (var);
	}
      else if ((QDPLL_VAR_POS_MARKED (var) && lit < 0) ||
	       (QDPLL_VAR_NEG_MARKED (var) && lit > 0))
	{			/* Clause is tautological */
	  unmark_clause_variables (qdpll, clause);
	  return clause;
	}
      else
	{			/* Clause contains multiple literals */
	  assert ((QDPLL_VAR_POS_MARKED (var) && lit > 0) ||
		  (QDPLL_VAR_NEG_MARKED (var) && lit < 0));
	  *p-- = *last--;
	  end--;
	  clause->num_lits--;
	}
    }

  unmark_clause_variables (qdpll, clause);
  QDPLL_SORT (qdpll, int, compare_lits_by_variable_nesting, clause->lits,
	      clause->num_lits);
  /* Catch parsed, empty clause or empty clause after universal reduction */
  if (clause->num_lits == 0 || universally_reduce_clause (qdpll, clause))
    {
      qdpll->state.found_empty_clause = 1;
      return clause;
    }

  return 0;
}


/* Push clause on clause stack, update occ_lists. */
static void
import_clause (QDPLL * qdpll, Constraint * clause)
{
  assert (!clause->is_cube);
  QDPLLMemMan *mm = qdpll->mm;
  Var *vars = qdpll->pcnf.vars;
  LINK_LAST (qdpll->pcnf.clauses, clause, link);
  assert (qdpll->pcnf.clauses.cnt ==
	  count_constraints (&(qdpll->pcnf.clauses)));
  unsigned int offset = 0;
  LitID *p, *end;
  for (p = clause->lits, end = p + clause->num_lits; p < end; p++)
    {
      LitID lit = *p;
      assert ((VarID) LIT2VARID (lit) < qdpll->pcnf.size_vars);
      Var *var = LIT2VARPTR (vars, lit);
      assert (offset == ((unsigned int) (p - clause->lits)));
      occ_link_last (lit, QDPLL_LIT_NEG (lit) ?
		     &(var->neg_occ_clauses) : &(var->pos_occ_clauses),
		     clause, offset);
      offset++;
    }
}


static Constraint *
create_constraint (QDPLL * qdpll, unsigned int num_lits, int is_cube)
{
  QDPLLMemMan *mm = qdpll->mm;
  Constraint *result = qdpll_malloc (mm,
				     sizeof (Constraint) +
				     num_lits * sizeof (LitID));
#ifndef NDEBUG
  result->size_lits = num_lits;
#endif
  result->is_cube = is_cube;
  result->dep_init_level = qdpll->num_deps_init;
  result->occ_link = qdpll_malloc (mm, num_lits * sizeof (OccLink));
  result->num_lits = num_lits;
  result->rwatcher_pos = result->lwatcher_pos = QDPLL_INVALID_WATCHER_POS;
  return result;
}


static void
delete_constraint (QDPLL * qdpll, Constraint * constraint)
{
  QDPLLMemMan *mm = qdpll->mm;
  int num_lits = constraint->num_lits;
#ifndef NDEBUG
  num_lits = constraint->size_lits;
#endif
  qdpll_free (mm, constraint->occ_link, num_lits * sizeof (OccLink));
  qdpll_free (mm, constraint,
	      sizeof (Constraint) + num_lits * sizeof (LitID));
}


/* Add literals/variables to clause or scope */
static const char *
import_added_ids (QDPLL * qdpll)
{
  LitIDStack *add_stack = &(qdpll->add_stack);
  LitID id;
  LitID *sp = add_stack->start, *se = add_stack->top;

  if (qdpll->state.scope_opened)
    {				/* Import scope's variables */
      Scope *scope = qdpll->pcnf.scopes.last;
      /* Must not add to default scope */
      assert (scope->nesting > QDPLL_DEFAULT_SCOPE_NESTING);

      while (sp < se)
	{
	  id = *sp++;
	  assert (id != 0);
	  if (id < 0)
	    return "negative variable ID in scope!";
	  qdpll_adjust_vars (qdpll, id);
	  Var *vars = qdpll->pcnf.vars;
	  if (vars[id].id != 0)
	    return "variable already quantified!";
	  declare_and_init_variable (qdpll, scope, id);
	}

      qdpll->state.scope_opened = 0;
    }
  else
    {				/* Import clause's literals */
      unsigned int num_lits = QDPLL_COUNT_STACK (*add_stack);
      Constraint *clause = create_constraint (qdpll, num_lits, 0);
      LitID *p = clause->lits;
      while (sp < se)
	{
	  id = *sp++;
	  assert (id != 0);
	  VarID var_id = LIT2VARID (id);
	  qdpll_adjust_vars (qdpll, var_id);
	  Var *var = qdpll->pcnf.vars + var_id;

	  if (var->id == 0)
	    {
	      /* Declare var; (Q)DIMACS backward compatibility */
	      Scope *scope = qdpll->pcnf.scopes.first;
	      assert (QDPLL_SCOPE_EXISTS (scope));
	      assert (scope->nesting == QDPLL_DEFAULT_SCOPE_NESTING);
	      declare_and_init_variable (qdpll, scope, var_id);
	    }
	  *p++ = id;		/* Add lits to clause */
	}

      /* Next, sort and clean up clause, then update occ-stacks */
      if (!cleanup_clause (qdpll, clause))
	import_clause (qdpll, clause);
      else			/* Clause is tautological or empty -> remove */
	delete_constraint (qdpll, clause);
    }

  QDPLL_RESET_STACK (*add_stack);
  return 0;
}


static int
has_variable_active_occs_in_cubes (QDPLL * qdpll, Var * var,
				   OccList * occ_cubes)
{
  if (QDPLL_VAR_ASSIGNED (var))
    return 0;

  LitID lit = occ_cubes == &(var->neg_occ_cubes) ? -var->id : var->id;
  OccListIterator it;
  OLITER_INIT (it, occ_cubes->first, occ_cubes->foffset, lit);
  while (OLITER_CUR (it))
    {
      assert (!is_cube_satisfied (qdpll, OLITER_CUR (it)));
      if (!is_cube_empty (qdpll, OLITER_CUR (it)))
	return 1;
      OLITER_NEXT (it, lit);
    }

  return 0;
}


/* Variable 'var' was identified as pure in clauses. Check if this is
   also the case in learnt cubes. */
static int
is_var_pure_in_cubes (QDPLL * qdpll, Var * var,
		      const QDPLLAssignment implied_value)
{
  if (QDPLL_SCOPE_FORALL (var->scope))
    {
      if (implied_value == QDPLL_ASSIGNMENT_TRUE)
	{
	  if (has_variable_active_occs_in_cubes
	      (qdpll, var, &(var->pos_occ_cubes)))
	    return 0;
	}
      else
	{
	  if (has_variable_active_occs_in_cubes
	      (qdpll, var, &(var->neg_occ_cubes)))
	    return 0;
	}
    }
  else
    {
      assert (QDPLL_SCOPE_EXISTS (var->scope));
      if (implied_value == QDPLL_ASSIGNMENT_TRUE)
	{
	  if (has_variable_active_occs_in_cubes
	      (qdpll, var, &(var->neg_occ_cubes)))
	    return 0;
	}
      else
	{
	  if (has_variable_active_occs_in_cubes
	      (qdpll, var, &(var->pos_occ_cubes)))
	    return 0;
	}
    }
  return 1;
}


static int
clause_has_true_ignored_lit (QDPLL * qdpll, Constraint * c, VarID skip_varid,
			     unsigned int ignore_dlevel)
{
  assert (!c->is_cube);
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *e;
  for (p = c->lits, e = p + c->num_lits; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      assert (var->id);
      if (skip_varid == var->id)
	continue;
      if (var->decision_level <= ignore_dlevel &&
	  ((QDPLL_VAR_ASSIGNED_TRUE (var) && QDPLL_LIT_POS (lit)) ||
	   (QDPLL_VAR_ASSIGNED_FALSE (var) && QDPLL_LIT_NEG (lit))))
	return 1;
    }
  return 0;
}


/* Disable all clauses which are satisfied by var's assignment. These
   clauses will be ignored in the dependency manager. value == 1 means
   disabling clauses. */
static void
toggle_occurrence_clauses (QDPLL * qdpll, Var * var, unsigned int value)
{
  assert (value == 0 || value == 1);
  assert (var->assignment != QDPLL_ASSIGNMENT_UNDEF);
  assert (value == 1 || var->decision_level > 0);
  assert (var->decision_level <= 0);
  assert (value == 0 || !var->mark_clauses_disabled);
  assert (value == 1 || var->mark_clauses_disabled);
#ifndef NDEBUG
  var->mark_clauses_disabled = value;
#endif
  if (qdpll->options.verbosity > 1)
    fprintf (stderr,
	     "Toggle clauses at ignore-level=%d, val=%d, Var %d assigned %d.\n",
	     0, value, var->id, var->assignment);
  LitID lit;
  OccList *occs;
  if (QDPLL_VAR_ASSIGNED_FALSE (var))
    {
      lit = -var->id;
      occs = &(var->neg_occ_clauses);
    }
  else
    {
      lit = var->id;
      occs = &(var->pos_occ_clauses);
    }

  OccListIterator it;
  if (value == 1)
    {
      OLITER_INIT (it, occs->first, occs->foffset, lit);
      while (OLITER_CUR (it))
	{
	  if (!OLITER_CUR (it)->disabled)
	    qdpll->state.disabled_clauses++;
	  OLITER_CUR (it)->disabled = value;
	  OLITER_NEXT (it, lit);
	}
    }
  else
    {
      QDPLL_ABORT_QDPLL (1, "must not enable clauses!");
      assert (value == 0);
      OLITER_INIT (it, occs->first, occs->foffset, lit);
      while (OLITER_CUR (it))
	{
	  if (!clause_has_true_ignored_lit
	      (qdpll, OLITER_CUR (it), var->id, 0))
	    OLITER_CUR (it)->disabled = value;
	  OLITER_NEXT (it, lit);
	}
    }
}


static void
push_assigned_variable (QDPLL * qdpll, Var * var, QDPLLAssignment assignment,
			QDPLLVarMode mode)
{
  assert (mode > 0 && mode <= 4);
  assert (mode != QDPLL_VARMODE_UNDEF);
  assert (assignment != QDPLL_ASSIGNMENT_UNDEF);
  assert (var->assignment == QDPLL_ASSIGNMENT_UNDEF);
  assert (var->mode == QDPLL_VARMODE_UNDEF);
  assert (!QDPLL_VAR_ASSIGNED (var));
  assert (!QDPLL_VAR_MARKED_PROPAGATED (var));
  assert (var->decision_level == QDPLL_INVALID_DECISION_LEVEL);

  if ((QDPLL_SCOPE_EXISTS (var->scope)) || (QDPLL_SCOPE_FORALL (var->scope)))
    var->cached_assignment = assignment;

  var->mode = mode;
  var->assignment = assignment;

  assert (!(QDPLL_SCOPE_EXISTS (var->scope) && mode == QDPLL_VARMODE_UNIT)
	  || (var->antecedent && !var->antecedent->is_cube));
  assert (!(var->antecedent && !var->antecedent->is_cube)
	  || (QDPLL_SCOPE_EXISTS (var->scope) && mode == QDPLL_VARMODE_UNIT));
  assert (!(QDPLL_SCOPE_FORALL (var->scope) && mode == QDPLL_VARMODE_UNIT)
	  || (var->antecedent && var->antecedent->is_cube));
  assert (!(var->antecedent && var->antecedent->is_cube)
	  || (QDPLL_SCOPE_FORALL (var->scope) && mode == QDPLL_VARMODE_UNIT));
  assert (!var->antecedent || var->antecedent->is_reason);

  if (mode < 3)
    {
      assert (mode == QDPLL_VARMODE_UNIT || mode == QDPLL_VARMODE_PURE);
      var->decision_level = qdpll->state.decision_level;
    }
  else
    {
      assert (mode == QDPLL_VARMODE_LBRANCH || mode == QDPLL_VARMODE_RBRANCH);
      var->decision_level = ++qdpll->state.decision_level;
    }

#ifndef NDEBUG
#if QDPLL_ASSERT_FIND_IN_ASSIGNED_VARS
  assert (!find_in_assigned_vars (qdpll, var->id));
#endif
#endif

  /* Variable will be assigned in during BCP. */
  push_assigned_vars (qdpll, var->id);

  if (qdpll->options.verbosity > 1)
    {
      fprintf (stderr,
	       "push assigned var.: id=%d, type=%c(%d), dlevel=%d, val=%d, mode=%d\n",
	       var->id, QDPLL_SCOPE_EXISTS (var->scope) ? 'E' : 'A',
	       var->scope->nesting, var->decision_level, var->assignment,
	       var->mode);
    }
#ifndef NDEBUG
#if QDPLL_ASSERT_PUSHED_PURE_LITS
  if (mode == QDPLL_VARMODE_PURE)
    assert_pushed_pure_lits (qdpll);
#endif
#endif

}


static int
has_variable_active_occs_in_clauses (QDPLL * qdpll, Var * var,
				     OccList * occ_clauses, int check_prop)
{
  assert (!check_prop);
  if (QDPLL_VAR_ASSIGNED (var))
    return 0;

  LitID lit = occ_clauses == &(var->neg_occ_clauses) ? -var->id : var->id;

  OccListIterator it;
  OLITER_INIT (it, occ_clauses->first, occ_clauses->foffset, lit);
  while (OLITER_CUR (it))
    {
      assert (!OLITER_CUR (it)->is_cube);
      assert (qdpll->bcp_ptr != qdpll->assigned_vars_top
	      || !is_clause_empty (qdpll, OLITER_CUR (it)));

      if ((!check_prop && !is_clause_satisfied (qdpll, OLITER_CUR (it))) ||
	  (check_prop
	   && !is_clause_satisfied_by_prop_var (qdpll, OLITER_CUR (it))))
	return 1;
      OLITER_NEXT (it, lit);
    }

  return 0;
}


/* -------------------- START: LEARNING -------------------- */


static void
compact_lit_stack (LitIDStack * lit_stack, LitID * start, unsigned int del)
{
  assert (!start || start <= lit_stack->top);
  assert (!start || lit_stack->start <= start);
  /* 'del' is the number of zeroes to be deleted. */
  assert (del);

  LitID *p = 0, *t, *nextn = 0, *e;
  for (t = start ? start : lit_stack->start, e = lit_stack->top; t < e; t++)
    {
      if (!*t)
	{
	  if (!p)
	    p = t;
	}
      else
	{
	  if (p)
	    {
	      assert (!nextn);
	      nextn = t;
	      break;
	    }
	}
    }
  if (!p)
    {
      assert (!nextn);
      p = nextn = e;
    }
  else if (!nextn)
    {
      nextn = e;
    }
  assert (p && nextn);
  assert (lit_stack->start <= p);
  assert (!nextn || lit_stack->start <= nextn);
  assert (p <= nextn);
  assert (p >= e || !*p);
  assert (nextn >= e || *nextn);

  assert (e == lit_stack->top);
  LitID *n = nextn;
  for (; p < e; p++)
    {
      assert (p <= n);
      if (!*p)
	{
	  for (n = nextn; n < e; n++)
	    {
	      LitID nlit = *n;
	      if (nlit)
		{
		  *p = nlit;
		  *n = 0;
		  nextn = n + 1;
		  break;
		}
	    }
	}
    }
  assert (e == lit_stack->top);
  lit_stack->top = e - del;

#ifndef NDEBUG
#if QDPLL_ASSERT_COMPACT_LIT_STACK
  LitID *e2;
  for (p = lit_stack->start, e2 = lit_stack->top; p < e; p++)
    {
      if (p < e2)
	assert (*p);
      else
	assert (!*p);
    }
#endif
#endif
}


static void
type_reduce_unmark_members (QDPLL * qdpll, Var * check_var)
{
  Var **p, **e;
  for (p = check_var->type_red_members.start,
       e = check_var->type_red_members.top; p < e; p++)
    {
      assert (LEARN_VAR_MARKED (*p));
      LEARN_VAR_UNMARK (*p);
    }
}


static void
type_reduce_unmark_by_graph (QDPLL * qdpll, VarPtrStack * other_lits,
			     const QDPLLQuantifierType type)
{
  QDPLLDepManGeneric *dm = qdpll->dm;
  dm->mark_inverse_depending (dm, other_lits, type);
}


static void
type_reduce_by_std_deps_adv (QDPLL * qdpll, LitIDStack * lit_stack,
			     const QDPLLQuantifierType type)
{
  assert (QDPLL_COUNT_STACK (qdpll->wreason_a) == 0);
  assert (QDPLL_COUNT_STACK (qdpll->wreason_e) == 0);
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *e;
#ifndef NDEBUG
  assert_lits_no_holes (qdpll, lit_stack->start, lit_stack->top);
  /* Lits must be non-empty and reduced. */
  assert (QDPLL_COUNT_STACK (*lit_stack) != 0);
#endif
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  unsigned int ignore_dlevel = 0, del = 0;
  assert (ignore_dlevel == 0);
  assert (qdpll->dep_check_cnt <=
	  ((unsigned int) QDPLL_COUNT_STACK (*lit_stack)));

  /* Fill separate lit-stacks. */
  QDPLLMemMan *mm = qdpll->mm;
  for (p = lit_stack->start, e = lit_stack->top; p < e; p++)
    {
      LitID lit = *p;
      assert (lit);
      Var *var = LIT2VARPTR (vars, lit);
      assert (var->decision_level > 0);
      assert (LEARN_VAR_MARKED (var));
      assert (QDPLL_LIT_POS (lit) || LEARN_VAR_NEG_MARKED (var));
      assert (QDPLL_LIT_NEG (lit) || LEARN_VAR_POS_MARKED (var));
      assert (!(LEARN_VAR_POS_MARKED (var) && LEARN_VAR_NEG_MARKED (var)));
      if (QDPLL_SCOPE_FORALL (var->scope))
	{

	  Var *rep = VARID2VARPTR (vars,
				   qdpll->dm->get_class_rep (qdpll->dm,
							     var->id, 0));
	  if (!QDPLL_VAR_POS_MARKED (rep))
	    {
	      QDPLL_VAR_POS_MARK (rep);
	      assert (QDPLL_COUNT_STACK (rep->type_red_members) == 0);
	      QDPLL_PUSH_STACK (mm, qdpll->wreason_a, rep);
	    }
	  /* Collect class members. */
	  QDPLL_PUSH_STACK (mm, rep->type_red_members, var);
	}
      else
	{
	  Var *rep = type == QDPLL_QTYPE_EXISTS ? VARID2VARPTR (vars,
								qdpll->dm->
								get_class_rep
								(qdpll->dm,
								 var->id,
								 1)) :
	    VARID2VARPTR (vars,
			  qdpll->dm->get_class_rep (qdpll->dm,
						    var->id,
						    0));
	  if (!QDPLL_VAR_POS_MARKED (rep))
	    {
	      QDPLL_VAR_POS_MARK (rep);
	      assert (QDPLL_COUNT_STACK (rep->type_red_members) == 0);
	      QDPLL_PUSH_STACK (mm, qdpll->wreason_e, rep);
	    }
	  /* Collect class members. */
	  QDPLL_PUSH_STACK (mm, rep->type_red_members, var);
	}
    }

  qdpll->dep_check_cnt = 0;

  VarPtrStack *check_lits, *other_lits;
  if (type == QDPLL_QTYPE_EXISTS)
    {
      check_lits = &(qdpll->wreason_e);
      other_lits = &(qdpll->wreason_a);
    }
  else
    {
      check_lits = &(qdpll->wreason_a);
      other_lits = &(qdpll->wreason_e);
    }

  Var **cp = check_lits->start, **ce = check_lits->top;
  Var **op = other_lits->start, **oe = other_lits->top;

  type_reduce_unmark_by_graph (qdpll, other_lits, type);
  for (; cp < ce; cp++)
    {
      Var *check_var = *cp;
      if (QDPLL_VAR_POS_MARKED (check_var))
	{
	  /* Whole class is reducible. */
	  type_reduce_unmark_members (qdpll, check_var);
	  del += QDPLL_COUNT_STACK (check_var->type_red_members);
	}
    }

  /* Finally traverse actual working reason and clean up */
  if (del)
    {
      for (p = lit_stack->start, e = lit_stack->top; p < e; p++)
	{
	  LitID lit = *p;
	  assert (lit);
	  Var *var = LIT2VARPTR (vars, lit);
	  if (!LEARN_VAR_MARKED (var))
	    {
	      *p = 0;
	      if (qdpll->options.verbosity > 1)
		{
		  if (type == QDPLL_QTYPE_FORALL)
		    fprintf (stderr,
			     "CDCL: type-reducing lit %d by std-deps\n", lit);
		  else
		    fprintf (stderr,
			     "SDCL: type-reducing lit %d by std-deps\n", lit);
		}
	    }
	}
      compact_lit_stack (lit_stack, 0, del);
    }

  /* Reset class representatives. */
  for (cp = check_lits->start; cp < ce; cp++)
    {
      QDPLL_VAR_UNMARK (*cp);
      QDPLL_RESET_STACK ((*cp)->type_red_members);
    }
  for (cp = other_lits->start; cp < oe; cp++)
    {
      QDPLL_VAR_UNMARK (*cp);
      QDPLL_RESET_STACK ((*cp)->type_red_members);
    }

  QDPLL_RESET_STACK (qdpll->wreason_a);
  QDPLL_RESET_STACK (qdpll->wreason_e);

#ifndef NDEBUG
  assert_lits_no_holes (qdpll, lit_stack->start, lit_stack->top);
#endif
}


static void
type_reduce (QDPLL * qdpll, LitIDStack * lit_stack,
	     const QDPLLQuantifierType other_type)
{
#ifndef NDEBUG
  assert_lits_no_holes (qdpll, lit_stack->start, lit_stack->top);
#endif
  assert (other_type == QDPLL_QTYPE_FORALL
	  || other_type == QDPLL_QTYPE_EXISTS);
  /* First reduce literals by scope order, then by standard-dependency scheme. */
  Var *lit_var, *vars = qdpll->pcnf.vars;
  LitID lit, *elemp, *start = lit_stack->start;
  while (start <= (elemp = lit_stack->top - 1))
    {
      lit = *elemp;
      lit_var = LIT2VARPTR (vars, lit);

      type_reduce_by_std_deps_adv (qdpll, lit_stack,
				   other_type ==
				   QDPLL_QTYPE_EXISTS ? QDPLL_QTYPE_FORALL :
				   QDPLL_QTYPE_EXISTS);
      break;
    }
}


/* Returns the number of existential literals 
   at 'level' in the working clause. */
static unsigned int
count_type_lit_at_dec_level (QDPLL * qdpll, LitID * lit_start,
			     LitID * lit_end, unsigned int level,
			     const QDPLLQuantifierType type)
{
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  assert (lit_start < lit_end);
  assert (level != QDPLL_INVALID_DECISION_LEVEL);
  Var *vars = qdpll->pcnf.vars;
  unsigned int cnt = 0;

  LitID *p, *e;
  for (p = lit_start, e = lit_end; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      assert (type != var->scope->type ||
	      var->decision_level != QDPLL_INVALID_DECISION_LEVEL);
      if (type == var->scope->type && var->decision_level == level)
	cnt++;
    }

  return cnt;
}


/* Assumes that clause is sorted. */
static unsigned int
get_reason_asserting_level (QDPLL * qdpll, LitID * lit_start, LitID * lit_end,
			    Var * implied_var, const QDPLLQuantifierType type)
{
  assert (lit_start < lit_end);
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  assert (type == implied_var->scope->type);
  Var *vars = qdpll->pcnf.vars;
  unsigned int level, highest = 0, next_highest = 0;
  QDPLLDepManGeneric *dm = qdpll->dm;

  LitID *p, *e;
  for (e = lit_start, p = lit_end - 1; e <= p; p--)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      level = var->decision_level;

      if (type != var->scope->type
	  && !dm->depends (dm, var->id, implied_var->id))
	continue;

      if (level > highest)
	{
	  assert (level != QDPLL_INVALID_DECISION_LEVEL);
	  next_highest = highest;
	  highest = level;
	}
      else if (level > next_highest)
	{
	  assert (level != QDPLL_INVALID_DECISION_LEVEL);
	  next_highest = level;
	}
    }

  return next_highest;
}


/* Returns the highest decision level of a 
   'type'-literal in the working reason. */
static unsigned int
get_highest_type_lit_dec_level (QDPLL * qdpll, LitID * lit_start,
				LitID * lit_end,
				const QDPLLQuantifierType type)
{
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  assert (lit_start < lit_end);
  Var *vars = qdpll->pcnf.vars;
  unsigned int level, max_level = 0;

  LitID *p, *e;
  for (p = lit_start, e = lit_end; p < e; p++)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      if (type == var->scope->type)
	{
	  assert (var->decision_level != QDPLL_INVALID_DECISION_LEVEL);
	  if ((level = var->decision_level) > max_level)
	    max_level = level;
	}
    }

  return max_level;
}


/* Returns variable at assigned at decision level 'level'. Note that
   the result is unique if, and only if function
   'count_exist_lit_at_dec_level' has returned 1 on that level. */
static Var *
get_type_var_at_dec_level (QDPLL * qdpll, LitID * lit_start, LitID * lit_end,
			   unsigned int level, const QDPLLQuantifierType type)
{
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  assert (lit_start < lit_end);
  Var *vars = qdpll->pcnf.vars;
  Var *var = 0;

  LitID *p, *e;
  for (p = lit_start, e = lit_end; p < e; p++)
    {
      LitID lit = *p;
      var = LIT2VARPTR (vars, lit);
      if (type == var->scope->type)
	{
	  assert (var->decision_level != QDPLL_INVALID_DECISION_LEVEL);
	  if (var->decision_level == level)
	    return var;
	}
    }

  assert (0);
  return 0;
}


static void
learnt_constraint_mtf (QDPLL * qdpll, Constraint * c)
{
  if (!c->learnt)
    return;
  if (c->is_cube)
    {
      UNLINK (qdpll->pcnf.learnt_cubes, c, link);
      LINK_FIRST (qdpll->pcnf.learnt_cubes, c, link);
    }
  else
    {
      UNLINK (qdpll->pcnf.learnt_clauses, c, link);
      LINK_FIRST (qdpll->pcnf.learnt_clauses, c, link);
    }
}


static void
decay_var_activity (QDPLL * qdpll)
{
  qdpll->state.var_act_inc *= (double) (1.0 / 0.95);
}


static void
increase_var_activity (QDPLL * qdpll, Var * var)
{
  assert (1 + var->scope->nesting);
  var->priority +=
    (qdpll->state.var_act_inc * (1 + ((double) var->scope->nesting) / 10));
  if (var->priority > 1e100)
    {
      Var *p, *e;
      for (p = qdpll->pcnf.vars, e = p + qdpll->pcnf.size_vars; p < e; p++)
	{
	  if (p->id)
	    p->priority *= 1e-100;
	}
      qdpll->state.var_act_inc *= 1e-100;
    }

  if (var->priority_pos != QDPLL_INVALID_PQUEUE_POS)
    var_pqueue_increase_key (qdpll, var->id);
}


static void
get_cover_collect_lit (QDPLL * qdpll, QDPLLMemMan * mm,
		       LitIDStack * lit_stack, Var * var, LitID lit)
{
  assert (var);
  assert (lit);
  assert (qdpll->pcnf.scopes.last != var->scope);
  assert ((QDPLL_LIT_NEG (lit) && QDPLL_VAR_ASSIGNED_FALSE (var)) ||
	  (QDPLL_LIT_POS (lit) && QDPLL_VAR_ASSIGNED_TRUE (var)));
  assert (!LEARN_VAR_MARKED (var));
  if (QDPLL_LIT_POS (lit))
    LEARN_VAR_POS_MARK (var);
  else
    LEARN_VAR_NEG_MARK (var);
  QDPLL_PUSH_STACK (mm, *lit_stack, lit);
}


static void
get_cover (QDPLL * qdpll, LitIDStack * lit_stack)
{
  QDPLLMemMan *mm = qdpll->mm;
  Var *vars = qdpll->pcnf.vars;
  assert (QDPLL_COUNT_STACK (*lit_stack) == 0);
  const Scope *last_scope = qdpll->pcnf.scopes.last;
  assert (QDPLL_SCOPE_EXISTS (last_scope));
  Constraint *c;
  for (c = qdpll->pcnf.clauses.first; c; c = c->link.next)
    {
      assert (!c->learnt);
      assert (!c->is_cube);
      if (c->disabled)
	continue;
      LitID *p, *e, lit;
      Var *max_e_true_var = 0, *min_a_true_var = 0;
      LitID max_e_true_lit = 0, min_a_true_lit = 0;
      for (p = c->lits, e = p + c->num_lits; p < e; p++)
	{
	  lit = *p;
	  Var *lit_var = LIT2VARPTR (vars, lit);
	  if (QDPLL_SCOPE_FORALL (lit_var->scope)
	      && lit_var->mode == QDPLL_VARMODE_PURE)
	    continue;
	  /* Search for positive literals. */
	  if ((QDPLL_LIT_NEG (lit) && QDPLL_VAR_ASSIGNED_FALSE (lit_var)) ||
	      (QDPLL_LIT_POS (lit) && QDPLL_VAR_ASSIGNED_TRUE (lit_var)))
	    {
	      assert (lit_var->decision_level > 0);
	      if (LEARN_VAR_MARKED (lit_var) || lit_var->scope == last_scope)
		goto SKIP;
	      if (QDPLL_SCOPE_FORALL (lit_var->scope))
		{
		  if (!min_a_true_var
		      || lit_var->scope->nesting <
		      min_a_true_var->scope->nesting)
		    {
		      min_a_true_var = lit_var;
		      min_a_true_lit = lit;
		    }
		}
	      else
		{
		  if (!max_e_true_var
		      || max_e_true_var->scope->nesting <
		      lit_var->scope->nesting)
		    {
		      max_e_true_var = lit_var;
		      max_e_true_lit = lit;
		    }
		}
	    }
	}
      assert (max_e_true_var || min_a_true_var);
      assert (!max_e_true_var || max_e_true_lit);
      assert (!min_a_true_var || min_a_true_lit);
      if (max_e_true_var)
	get_cover_collect_lit (qdpll, mm, lit_stack, max_e_true_var,
			       max_e_true_lit);
      else
	get_cover_collect_lit (qdpll, mm, lit_stack, min_a_true_var,
			       min_a_true_lit);
    SKIP:;
    }
  qdpll->dep_check_cnt = QDPLL_COUNT_STACK (*lit_stack);
}


/* Initialize the literal-stack with literals of either the
   conflicting clause or of a cover set / satisfied cube. This is the
   working reason to start with. */
static void
get_initial_reason (QDPLL * qdpll, LitIDStack * lit_stack,
		    const QDPLLQuantifierType type)
{
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  Var *vars = qdpll->pcnf.vars;
  QDPLLMemMan *mm = qdpll->mm;
  assert (qdpll->dep_check_cnt == 0);
  Var *var;
  LitID *p, *e, lit;
  Constraint *res_cons = qdpll->result_constraint;
  if (type == QDPLL_QTYPE_EXISTS || (res_cons))
    {
      assert (type != QDPLL_QTYPE_EXISTS || (res_cons && !res_cons->is_cube));
      assert (type == QDPLL_QTYPE_EXISTS || (res_cons && res_cons->is_cube));
      assert (res_cons->dep_init_level <= qdpll->num_deps_init);
      learnt_constraint_mtf (qdpll, res_cons);
      /* Push and mark literals of reason clause onto 'lit-stack',
         which is the working reason. */
      p = res_cons->lits;
      e = p + res_cons->num_lits;
      assert (p < e);
      for (; p < e; p++)
	{
	  lit = *p;
	  var = LIT2VARPTR (vars, lit);
	  increase_var_activity (qdpll, var);
	  assert (!LEARN_VAR_MARKED (var));
	  if (var->decision_level == 0)
	    {
	      if (qdpll->options.verbosity > 1)
		{
		  if (type == QDPLL_QTYPE_EXISTS)
		    fprintf (stderr,
			     "\nCDCL: discarding top-level assignment: %d\n",
			     var->id);
		  else
		    fprintf (stderr,
			     "\nSDCL: discarding top-level assignment: %d\n",
			     var->id);
		}
	      continue;
	    }
	  if (QDPLL_LIT_NEG (lit))
	    LEARN_VAR_NEG_MARK (var);
	  else
	    LEARN_VAR_POS_MARK (var);
	  QDPLL_PUSH_STACK (mm, *lit_stack, lit);
	}
      if (res_cons->dep_init_level < qdpll->num_deps_init)
	qdpll->dep_check_cnt = QDPLL_COUNT_STACK (*lit_stack);
    }
  else
    {
      assert (type == QDPLL_QTYPE_FORALL && !res_cons);
      if (qdpll->options.verbosity > 1)
	fprintf (stderr, "SDCL: generating cover set.\n");
      /* Find cover set. */
      get_cover (qdpll, lit_stack);
    }

  type_reduce (qdpll, lit_stack, type);

  if (type == QDPLL_QTYPE_FORALL)
    {
      unsigned int cnt = QDPLL_COUNT_STACK (*lit_stack);
      QDPLL_SORT (qdpll, int, compare_lits_by_variable_nesting,
		  lit_stack->start, cnt);
    }
}


/* Returns true if variable is assigned at a level which has
   an existential decision variable. */
static int
is_var_at_type_dec_level (QDPLL * qdpll, Var * var,
			  const QDPLLQuantifierType type)
{
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  Var *vars = qdpll->pcnf.vars;

  VarID *p, *e;
  for (p = qdpll->assigned_vars_top - 1, e = qdpll->assigned_vars;
       p >= e; p--)
    {
      Var *v = VARID2VARPTR (vars, *p);
      if (v == var)
	{
	  VarID *p1;
	  for (p1 = p; p1 >= e; p1--)
	    {
	      v = VARID2VARPTR (vars, *p1);
	      if (v->decision_level < var->decision_level)
		return 0;
	      assert (v->decision_level == var->decision_level);
	      if (type == v->scope->type &&
		  (v->mode == QDPLL_VARMODE_LBRANCH ||
		   v->mode == QDPLL_VARMODE_RBRANCH))
		return 1;
	    }
	  return 0;
	}
    }

  return 0;
}


/* Assumes that the clause is sorted and that 'other_type_var' is the
   only type-literal at the maximal type-decision level. */
static int
all_smaller_type_lits_have_value (QDPLL * qdpll, LitIDStack * lit_stack,
				  Var * other_type_var,
				  const QDPLLQuantifierType type)
{
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  assert (other_type_var->scope->type != type);
  QDPLLDepManGeneric *dm = qdpll->dm;
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *e;
  for (p = lit_stack->top - 1, e = lit_stack->start; e <= p; p--)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      if (var == other_type_var)
	{
	  /* Check "smaller" type-variables. */
	  for (p--; e <= p; p--)
	    {
	      lit = *p;
	      var = LIT2VARPTR (vars, lit);
	      if (type == var->scope->type)
		{
		  if (!dm->depends (dm, var->id, other_type_var->id))
		    continue;

		  if (!QDPLL_VAR_ASSIGNED (var) ||
		      var->decision_level >= other_type_var->decision_level ||
		      (type == QDPLL_QTYPE_FORALL &&
		       ((QDPLL_VAR_ASSIGNED_FALSE (var)
			 && QDPLL_LIT_NEG (lit))
			|| (QDPLL_VAR_ASSIGNED_TRUE (var)
			    && QDPLL_LIT_POS (lit))))
		      || (type == QDPLL_QTYPE_EXISTS
			  &&
			  ((QDPLL_VAR_ASSIGNED_FALSE (var)
			    && QDPLL_LIT_POS (lit))
			   || (QDPLL_VAR_ASSIGNED_TRUE (var)
			       && QDPLL_LIT_NEG (lit)))))
		    return 0;
		}
	    }
	  return 1;
	}
    }
  assert (0);
  return 0;
}


static void
check_marks_and_push (QDPLL * qdpll, Var * var, LitID lit, LitIDStack * stack,
		      const QDPLLQuantifierType type)
{
  if (QDPLL_LIT_NEG (lit))
    {
      if (!LEARN_VAR_MARKED (var))
	{
	  LEARN_VAR_NEG_MARK (var);
	  QDPLL_PUSH_STACK (qdpll->mm, *stack, lit);
	  increase_var_activity (qdpll, var);
	}
      else if (LEARN_VAR_POS_MARKED (var))
	{
	  assert (type != var->scope->type);
	  if (!LEARN_VAR_NEG_MARKED (var))
	    {
	      LEARN_VAR_NEG_MARK (var);
	      QDPLL_PUSH_STACK (qdpll->mm, *stack, lit);
	      increase_var_activity (qdpll, var);
	    }
	}
    }
  else
    {
      assert (QDPLL_LIT_POS (lit));
      if (!LEARN_VAR_MARKED (var))
	{
	  LEARN_VAR_POS_MARK (var);
	  QDPLL_PUSH_STACK (qdpll->mm, *stack, lit);
	  increase_var_activity (qdpll, var);
	}
      else if (LEARN_VAR_NEG_MARKED (var))
	{
	  assert (type != var->scope->type);
	  if (!LEARN_VAR_POS_MARKED (var))
	    {
	      LEARN_VAR_POS_MARK (var);
	      QDPLL_PUSH_STACK (qdpll->mm, *stack, lit);
	      increase_var_activity (qdpll, var);
	    }
	}
    }
}


/* Perform q-resolution. */
static void
resolve_and_reduce (QDPLL * qdpll, LitIDStack ** lit_stack,
		    LitIDStack ** lit_stack_tmp, LitID * other_lits_start,
		    LitID * other_lits_end, Var * var,
		    const int full_reduce_by_deps,
		    const QDPLLQuantifierType type)
{
  assert (*lit_stack != *lit_stack_tmp);
  assert (*lit_stack != &(qdpll->add_stack)
	  || *lit_stack_tmp == &(qdpll->add_stack_tmp));
  assert (*lit_stack_tmp != &(qdpll->add_stack)
	  || *lit_stack == &(qdpll->add_stack_tmp));
  assert (other_lits_start < other_lits_end);
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
#ifndef NDEBUG
  assert_lits_sorted (qdpll, (*lit_stack)->start, (*lit_stack)->top);
  assert_lits_no_holes (qdpll, (*lit_stack)->start, (*lit_stack)->top);
#endif
  if (qdpll->options.verbosity > 1)
    {
      if (type == QDPLL_QTYPE_EXISTS)
	{
	  fprintf (stderr, "\nCDCL: pivot variable: %d\n", var->id);
	  fprintf (stderr, "CDCL: working clause: ");
	  print_lits (qdpll, (*lit_stack)->start,
		      QDPLL_COUNT_STACK (**lit_stack));
	  fprintf (stderr, "CDCL: antecedent: ");
	  print_lits (qdpll, other_lits_start,
		      other_lits_end - other_lits_start);
	}
      else
	{
	  fprintf (stderr, "\nSDCL: pivot variable: %d\n", var->id);
	  fprintf (stderr, "SDCL: working cube: ");
	  print_lits (qdpll, (*lit_stack)->start,
		      QDPLL_COUNT_STACK (**lit_stack));
	  fprintf (stderr, "SDCL: antecedent: ");
	  print_lits (qdpll, other_lits_start,
		      other_lits_end - other_lits_start);
	}
    }

  QDPLLMemMan *mm = qdpll->mm;
  Var *vars = qdpll->pcnf.vars;
  unsigned int del = 0;
  LitIDStack *tmp = *lit_stack_tmp;
  assert (QDPLL_COUNT_STACK (*tmp) == 0);
  assert (QDPLL_COUNT_STACK (**lit_stack) != 0);
  LitID *p1, *p2;
  const LitID *e1 = (*lit_stack)->top;
  const LitID *e2 = other_lits_end;
  LitID lit1, lit2;
  Var *var1, *var2;
  VarID vid1;
  VarID vid2;
  unsigned int nesting1;
  unsigned int nesting2;

  assert (qdpll->dep_check_cnt == 0);

  p1 = (*lit_stack)->start;
  p2 = other_lits_start;
  assert (p1 < e1);
  assert (p2 < e2);
  lit1 = *p1;
  lit2 = *p2;
#ifndef NDEBUG
  int wreason_seen_pivot = 0;
#endif
  /* Merge sorted lists */
  while (1)
    {
      if (compare_lits_by_variable_nesting (qdpll, lit1, lit2) <= 0)
	{
	  var1 = LIT2VARPTR (vars, lit1);
	  assert (var1->decision_level > 0);
	  assert (QDPLL_LIT_NEG (lit1) || LEARN_VAR_POS_MARKED (var1));
	  assert (QDPLL_LIT_POS (lit1) || LEARN_VAR_NEG_MARKED (var1));
	  /* Must ignore pivot variable. */
	  if (var1 != var)
	    QDPLL_PUSH_STACK (mm, *tmp, lit1);
	  else
	    {
#ifndef NDEBUG
	      wreason_seen_pivot = 1;
#endif
	      LEARN_VAR_UNMARK (var1);
	    }
	  p1++;
	  if (p1 >= e1)
	    break;
	  lit1 = *p1;
	}
      else
	{
	  var2 = LIT2VARPTR (vars, lit2);
	  assert (var2 != var || var2->decision_level > 0);
	  assert (var2 != var
		  || (wreason_seen_pivot && (*lit_stack)->start < p1
		      && LIT2VARPTR (vars, *(p1 - 1)) == var));
	  if (var2 == var)
	    {
	      if (!full_reduce_by_deps)
		qdpll->dep_check_cnt = QDPLL_COUNT_STACK (*tmp);
	    }
	  else if (var2->decision_level > 0)
	    check_marks_and_push (qdpll, var2, lit2, tmp, type);
	  p2++;
	  if (p2 >= e2)
	    break;
	  lit2 = *p2;
	}
    }
  assert (p1 >= e1 || p2 >= e2);

  while (p1 < e1)
    {
      lit1 = *p1;
      var1 = LIT2VARPTR (vars, lit1);
      assert (var1->decision_level > 0);
      assert (QDPLL_LIT_NEG (lit1) || LEARN_VAR_POS_MARKED (var1));
      assert (QDPLL_LIT_POS (lit1) || LEARN_VAR_NEG_MARKED (var1));
      /* Must ignore pivot variable. */
      if (var1 != var)
	QDPLL_PUSH_STACK (mm, *tmp, lit1);
      else
	{
#ifndef NDEBUG
	  wreason_seen_pivot = 1;
#endif
	  LEARN_VAR_UNMARK (var1);
	}
      p1++;
    }

  while (p2 < e2)
    {
      lit2 = *p2;
      var2 = LIT2VARPTR (vars, lit2);
      assert (var2 != var || var2->decision_level > 0);
      assert (var2 != var
	      || (wreason_seen_pivot && (*lit_stack)->start < p1
		  && LIT2VARPTR (vars, *(p1 - 1)) == var));
      if (var2 == var)
	{
	  if (!full_reduce_by_deps)
	    qdpll->dep_check_cnt = QDPLL_COUNT_STACK (*tmp);
	}
      else if (var2->decision_level > 0)
	check_marks_and_push (qdpll, var2, lit2, tmp, type);
      p2++;
    }

  if (full_reduce_by_deps)
    qdpll->dep_check_cnt = QDPLL_COUNT_STACK (*tmp);

  /* Swap stacks. */
  LitIDStack *swap_tmp = *lit_stack;
  assert (tmp == *lit_stack_tmp);
  *lit_stack = tmp;
  *lit_stack_tmp = swap_tmp;
  QDPLL_RESET_STACK (**lit_stack_tmp);

#ifndef NDEBUG
  assert_lits_sorted (qdpll, (*lit_stack)->start, (*lit_stack)->top);
  assert_lits_no_holes (qdpll, (*lit_stack)->start, (*lit_stack)->top);
#endif

  type_reduce (qdpll, *lit_stack, type);
}


/* Checks whether the generated constraint on 
   'lit_stack' is asserting. */
static int
stop_resolution (QDPLL * qdpll, LitIDStack * lit_stack,
		 const QDPLLQuantifierType type)
{
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  unsigned int max_type_level =
    get_highest_type_lit_dec_level (qdpll, lit_stack->start, lit_stack->top,
				    type);
  Var *type_var;

  if (count_type_lit_at_dec_level
      (qdpll, lit_stack->start, lit_stack->top, max_type_level, type) != 1)
    return 0;
  else
    if (!is_var_at_type_dec_level
	(qdpll,
	 (type_var =
	  get_type_var_at_dec_level (qdpll, lit_stack->start, lit_stack->top,
				     max_type_level, type)), type))
    return 0;
  else if (!all_smaller_type_lits_have_value (qdpll, lit_stack, type_var,
					      type == QDPLL_QTYPE_EXISTS ?
					      QDPLL_QTYPE_FORALL :
					      QDPLL_QTYPE_EXISTS))
    return 0;
  else
    return 1;
}


/* Check if resolving 'lit_stack' with antecedent of 'var' would
   produce a constraint containing complementary literals. In this
   case resolution is blocked. */
static Var *
peek_tautology (QDPLL * qdpll, LitIDStack * lit_stack, Var * var)
{
  assert (var->antecedent);
  Var *vars = qdpll->pcnf.vars;
  Var *lit_var;
  LitID *p, *e, lit;
  for (p = var->antecedent->lits, e = p + var->antecedent->num_lits; p < e;
       p++)
    {
      lit = *p;
      lit_var = LIT2VARPTR (vars, lit);
      /* Ignore the pivot variable. */
      if (var == lit_var)
	continue;
      if ((QDPLL_LIT_NEG (lit) && LEARN_VAR_POS_MARKED (lit_var)) ||
	  (QDPLL_LIT_POS (lit) && LEARN_VAR_NEG_MARKED (lit_var)))
	{
	  if (qdpll->options.verbosity > 1)
	    {
	      fprintf (stderr, "peek tautology: tested var is %d\n", var->id);
	      fprintf (stderr, "peek tautology: lit stack is\n");
	      print_lits (qdpll, lit_stack->start,
			  lit_stack->top - lit_stack->start);
	      fprintf (stderr, "peek tautology: ante. is\n");
	      print_lits (qdpll, var->antecedent->lits,
			  var->antecedent->num_lits);

	      fprintf (stderr, "peek tautology: true by lit %d\n", lit);
	    }
	  return lit_var;
	}
    }
  return 0;
}


static void
choose_var_cleanup_classes (QDPLL * qdpll)
{
  Var **p, **e;
  for (p = qdpll->choose_var_aclasses.start,
       e = qdpll->choose_var_aclasses.top; p < e; p++)
    {
      assert ((*p)->choose_var_class_collected);
      (*p)->choose_var_class_collected = 0;
    }
  for (p = qdpll->choose_var_eclasses.start,
       e = qdpll->choose_var_eclasses.top; p < e; p++)
    {
      assert ((*p)->choose_var_class_collected);
      (*p)->choose_var_class_collected = 0;
    }
  QDPLL_RESET_STACK (qdpll->choose_var_aclasses);
  QDPLL_RESET_STACK (qdpll->choose_var_eclasses);
}


/* Choose pivot variable. */
static VarID
choose_var (QDPLL * qdpll, LitIDStack * lit_stack, VarID ** trail_ptr,
	    const QDPLLQuantifierType type)
{
  assert (QDPLL_COUNT_STACK (qdpll->choose_var_aclasses) == 0);
  assert (QDPLL_COUNT_STACK (qdpll->choose_var_eclasses) == 0);
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  assert (QDPLL_COUNT_STACK (*lit_stack) != 0);
  Var *blocking = 0, *pivot = 0, *max_trail_var = 0, *max_scope_var =
    0, *var, *vars = qdpll->pcnf.vars;
  LitID *p, *e, lit;
  QDPLLDepManGeneric *dm = qdpll->dm;
  QDPLLMemMan *mm = qdpll->mm;

  /* Get resolution candidates: collect all units. */
  for (p = lit_stack->top - 1, e = lit_stack->start; e <= p; p--)
    {
      lit = *p;
      var = LIT2VARPTR (vars, lit);

      /* Collect classes. */
      Var *rep =
	VARID2VARPTR (vars, qdpll->dm->get_class_rep (qdpll->dm, var->id, 0));
      if (!rep->choose_var_class_collected)
	{
	  rep->choose_var_class_collected = 1;
	  if (QDPLL_SCOPE_FORALL (rep->scope))
	    QDPLL_PUSH_STACK (mm, qdpll->choose_var_aclasses, rep);
	  else
	    QDPLL_PUSH_STACK (mm, qdpll->choose_var_eclasses, rep);
	}

      assert (LEARN_VAR_MARKED (var));
      if (var->scope->type == type && var->mode == QDPLL_VARMODE_UNIT)
	{
	  if (!var->mark_res_cand)
	    {
	      var->mark_res_cand = 1;
	      QDPLL_PUSH_STACK (qdpll->mm, qdpll->res_cands, var);
	    }
	}
    }

  if (QDPLL_EMPTY_STACK (qdpll->res_cands))
    {
      choose_var_cleanup_classes (qdpll);
      return 0;
    }

  unsigned int pivot_offset = 0;
  unsigned int max_trail_offset = 0;
  unsigned int max_scope_offset = 0;
  Var **vp, **ve;
  for (vp = qdpll->res_cands.start, ve = qdpll->res_cands.top; vp < ve; vp++)
    {
      Var *cand = *vp;
      assert (cand->scope->type == type);
      assert (cand->mode == QDPLL_VARMODE_UNIT);
      assert (LEARN_VAR_MARKED (cand));
      /* Find largest trail var. */
      if (!max_trail_var || max_trail_var->trail_pos < cand->trail_pos)
	{
	  max_trail_var = cand;
	  max_trail_offset = vp - qdpll->res_cands.start;
	}
    }
  assert (max_trail_var);
  assert (max_trail_offset <
	  ((unsigned int) (qdpll->res_cands.top - qdpll->res_cands.start)));
  assert (qdpll->res_cands.start[max_trail_offset] == max_trail_var);

  /* Check if trail-based pivot is blocked. */
  if ((blocking = peek_tautology (qdpll, lit_stack, max_trail_var)))
    {
      max_scope_offset = 0;
      max_scope_var = 0;
      for (vp = qdpll->res_cands.start, ve = qdpll->res_cands.top; vp < ve;
	   vp++)
	{
	  Var *cand = *vp;
	  assert (cand->scope->type == type);
	  assert (cand->mode == QDPLL_VARMODE_UNIT);
	  assert (LEARN_VAR_MARKED (cand));
	  int has_dep_in_lits = 1;
	  if ((!has_dep_in_lits ||
	       blocking->scope->nesting < cand->scope->nesting) &&
	      !peek_tautology (qdpll, lit_stack, cand) &&
	      (!max_scope_var || max_scope_var->trail_pos < cand->trail_pos))
	    {
	      max_scope_var = cand;
	      max_scope_offset = vp - qdpll->res_cands.start;
	    }
	}
      assert (max_scope_var);
      assert (max_scope_offset <
	      ((unsigned int) (qdpll->res_cands.top -
			       qdpll->res_cands.start)));
      assert (qdpll->res_cands.start[max_scope_offset] == max_scope_var);

      if (!max_scope_var)
	{
	  choose_var_cleanup_classes (qdpll);
	  return 0;
	}
      assert (max_scope_var && max_scope_var != max_trail_var);

      pivot = max_scope_var;
      pivot_offset = max_scope_offset;
    }
  else
    {
      pivot = max_trail_var;
      pivot_offset = max_trail_offset;
    }

  assert (pivot->mark_res_cand);
  /* Remove from cand-stack. */
  pivot->mark_res_cand = 0;
  Var *last = QDPLL_POP_STACK (qdpll->res_cands);
  qdpll->res_cands.start[pivot_offset] = last;

  choose_var_cleanup_classes (qdpll);
  return pivot->id;
}


/* Returns non-zero if a valid reason was derived 
   or 0 if either aborted or done. */
static int
generate_reason (QDPLL * qdpll, LitIDStack ** lit_stack,
		 LitIDStack ** lit_stack_tmp, VarID ** trail_ptr,
		 const QDPLLQuantifierType type)
{
  assert (!qdpll->state.abort_learning);
  assert (*lit_stack != *lit_stack_tmp);
  assert (*lit_stack != &(qdpll->add_stack)
	  || *lit_stack_tmp == &(qdpll->add_stack_tmp));
  assert (*lit_stack_tmp != &(qdpll->add_stack)
	  || *lit_stack == &(qdpll->add_stack_tmp));
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  QDPLLMemMan *mm = qdpll->mm;
  Var *vars = qdpll->pcnf.vars;
  VarID varid;

AGAIN:

  if (QDPLL_EMPTY_STACK (**lit_stack))
    return 0;

  int stop = stop_resolution (qdpll, *lit_stack, type);
  assert (!working_clause_is_tautologous (qdpll, *lit_stack, type));
  if (stop)
    return 1;
  else
    {
      /* Choose pivot. */
      varid = choose_var (qdpll, *lit_stack, trail_ptr, type);

      if (!varid)
	{
	  assert (!qdpll->state.abort_learning);
	  qdpll->state.abort_learning = 1;
	  return 0;
	}

      Var *var = VARID2VARPTR (vars, varid);
      LitID *other_lits_start, *other_lits_end;
      assert (type == QDPLL_QTYPE_FORALL
	      || (var->antecedent && !var->antecedent->is_cube));
      assert (type == QDPLL_QTYPE_EXISTS
	      || (var->antecedent && var->antecedent->is_cube));

      Constraint *antecedent = var->antecedent;
      assert (antecedent);
      assert (antecedent->is_reason);
      learnt_constraint_mtf (qdpll, antecedent);
      other_lits_start = antecedent->lits;
      other_lits_end = other_lits_start + antecedent->num_lits;

      resolve_and_reduce (qdpll, lit_stack, lit_stack_tmp, other_lits_start,
			  other_lits_end, var,
			  antecedent->dep_init_level < qdpll->num_deps_init,
			  type);
      /* Stack 'lit_stack' will now hold the resolvent's literals. */
      goto AGAIN;
    }
}


/* Set watcher to the literal which is implied at the asserting
   level (i.e. that is the literal with the highest decision level) and
   the other watcher to the second highest decision level. */
static void
set_learnt_constraint_lit_watchers (QDPLL * qdpll,
				    Constraint * learnt_constraint,
				    unsigned int asserting_level)
{
  Var *vars = qdpll->pcnf.vars;
  const int is_cube = learnt_constraint->is_cube;
  unsigned int hoffset = 0, nhoffset = 0, level;
  LitID *p, *e;
  for (e = learnt_constraint->lits, p = e + learnt_constraint->num_lits - 1;
       e <= p; p--)
    {
      LitID lit = *p;
      Var *var = LIT2VARPTR (vars, lit);
      level = var->decision_level;
      if (level > asserting_level &&
	  ((!is_cube && QDPLL_SCOPE_EXISTS (var->scope)) ||
	   (is_cube && QDPLL_SCOPE_FORALL (var->scope))))
	{
	  /* Get position of implied literal. */
	  assert (!hoffset);
	  hoffset = p - e;
	}

      if (level == asserting_level
	  && (((!is_cube && QDPLL_SCOPE_EXISTS (var->scope)) ||
	       (is_cube && QDPLL_SCOPE_FORALL (var->scope))) || hoffset))
	nhoffset = p - e;
    }

  assert (learnt_constraint->num_lits == 1 || hoffset != nhoffset);
  assert (nhoffset < learnt_constraint->num_lits);
  assert (hoffset < learnt_constraint->num_lits);

  unsigned int right_offset, left_offset;
  if (hoffset < nhoffset)
    {
      left_offset = hoffset;
      right_offset = nhoffset;
    }
  else
    {
      left_offset = nhoffset;
      right_offset = hoffset;
    }

  init_literal_watcher (qdpll, learnt_constraint, left_offset, right_offset);
}


/* Chronological backtracking. */
static unsigned int
chron_backtracking (QDPLL * qdpll, const QDPLLQuantifierType type)
{
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  assert (QDPLL_ASSIGNMENT_TRUE == -QDPLL_ASSIGNMENT_FALSE);
  assert (QDPLL_ASSIGNMENT_FALSE == -QDPLL_ASSIGNMENT_TRUE);
  VarID *p, *e, id;
  Var *vars = qdpll->pcnf.vars;
  for (p = qdpll->assigned_vars_top - 1, e = qdpll->assigned_vars; p >= e;
       p--)
    {
      id = *p;
      Var *assigned_var = VARID2VARPTR (vars, id);
      assert (QDPLL_VAR_ASSIGNED (assigned_var));
      assert (assigned_var->mode != QDPLL_VARMODE_UNDEF);
      if (type == assigned_var->scope->type
	  && assigned_var->mode == QDPLL_VARMODE_LBRANCH)
	{
	  assert (assigned_var->decision_level !=
		  QDPLL_INVALID_DECISION_LEVEL);
	  assert (!qdpll->state.forced_assignment.antecedent);
	  assert (!qdpll->state.forced_assignment.var);
	  assert (!qdpll->state.forced_assignment.assignment);
	  assert (!qdpll->state.forced_assignment.mode);
	  /* Set forced assignment (flipping decision variable) to be
	     enqueued afterwards. */
	  qdpll->state.forced_assignment.var = assigned_var;
	  qdpll->state.forced_assignment.assignment =
	    -assigned_var->assignment;
	  qdpll->state.forced_assignment.mode = QDPLL_VARMODE_RBRANCH;
	  return assigned_var->decision_level;
	}
    }

  return QDPLL_INVALID_DECISION_LEVEL;
}


static unsigned int
analyze_solution_no_sdcl (QDPLL * qdpll)
{
  return chron_backtracking (qdpll, QDPLL_QTYPE_FORALL);
}


static unsigned int
analyze_conflict_no_cdcl (QDPLL * qdpll)
{
  return chron_backtracking (qdpll, QDPLL_QTYPE_EXISTS);
}


static void
fill_sdcl_assigned_vars (QDPLL * qdpll)
{
  assert (QDPLL_COUNT_STACK (qdpll->sdcl_assigned_vars) == 0);
  QDPLLMemMan *mm = qdpll->mm;
  VarIDStack *assigned_stack = &(qdpll->sdcl_assigned_vars);
  VarID *p, *e;
  for (p = qdpll->assigned_vars, e = qdpll->bcp_ptr; p < e; p++)
    QDPLL_PUSH_STACK (mm, *assigned_stack, *p);
  qdpll->sdcl_uncovered_clauses = qdpll->pcnf.clauses.cnt;
  assert (qdpll->sdcl_uncovered_clauses >= qdpll->state.disabled_clauses);
  qdpll->sdcl_uncovered_clauses -= qdpll->state.disabled_clauses;
}


static unsigned int
generate_and_add_reason (QDPLL * qdpll, const QDPLLQuantifierType type)
{
  assert (type == QDPLL_QTYPE_FORALL || type == QDPLL_QTYPE_EXISTS);
  QDPLLMemMan *mm = qdpll->mm;
  Var *vars = qdpll->pcnf.vars;
  LitIDStack *lit_stack = &(qdpll->add_stack);
  LitIDStack *lit_stack_tmp = &(qdpll->add_stack_tmp);
  assert (QDPLL_COUNT_STACK (*lit_stack) == 0);
  assert (QDPLL_COUNT_STACK (*lit_stack_tmp) == 0);
  assert (QDPLL_COUNT_STACK (qdpll->sdcl_assigned_vars) == 0);
  assert (QDPLL_COUNT_STACK (qdpll->res_cands) == 0);
#ifndef NDEBUG
#if QDPLL_ASSERT_LEARN_VARS_UNMARKED
  assert_learn_vars_unmarked (qdpll);
#endif
#endif

  if (type == QDPLL_QTYPE_FORALL)
    fill_sdcl_assigned_vars (qdpll);

  get_initial_reason (qdpll, lit_stack, type);
  assert (type == QDPLL_QTYPE_FORALL || QDPLL_COUNT_STACK (*lit_stack) != 0);

  if (qdpll->options.verbosity > 1)
    {
      if (type == QDPLL_QTYPE_EXISTS)
	fprintf (stderr, "CDCL: conflicting clause: ");
      else
	fprintf (stderr, "SDCL: working reason: ");
      print_lits (qdpll, lit_stack->start, QDPLL_COUNT_STACK (*lit_stack));
    }

  /* Now lit-stack contains literals of either
     conflicting clause or cover-set/satisfied cube. */

  VarID *tmp_trail_ptr = qdpll->assigned_vars_top;
  int success =
    generate_reason (qdpll, &lit_stack, &lit_stack_tmp, &tmp_trail_ptr, type);
  assert (QDPLL_COUNT_STACK (*lit_stack_tmp) == 0);
  /* Trail-ptr has never been set. */
  assert (tmp_trail_ptr == qdpll->assigned_vars_top);
  tmp_trail_ptr--;

  LitID *p, *e, lit;
  Var *var;
  for (; tmp_trail_ptr < qdpll->assigned_vars_top; tmp_trail_ptr++)
    {
      VarID varid = *tmp_trail_ptr;
      var = VARID2VARPTR (vars, varid);
      if (LEARN_VAR_MARKED (var))
	{
	  assert (!success
		  || !(LEARN_VAR_POS_MARKED (var)
		       && LEARN_VAR_NEG_MARKED (var)));
	  LEARN_VAR_UNMARK (var);
	  Constraint *antecedent;
	  if ((antecedent = var->antecedent))
	    {
	      assert (antecedent->is_reason);
	      p = antecedent->lits;
	      e = p + antecedent->num_lits;
	      /* Unmark antecedent's variables. */
	      for (; p < e; p++)
		{
		  lit = *p;
		  var = LIT2VARPTR (vars, lit);
		  LEARN_VAR_UNMARK (var);
		}
	    }
	}
    }

  /* Unmark constraint variables. */
  for (p = lit_stack->start, e = lit_stack->top; p < e; p++)
    {
      lit = *p;
      var = LIT2VARPTR (vars, lit);
      assert (!success
	      || !(LEARN_VAR_POS_MARKED (var) && LEARN_VAR_NEG_MARKED (var)));
      LEARN_VAR_UNMARK (var);
    }
  Var **vp, **ve;
  for (vp = qdpll->res_cands.start, ve = qdpll->res_cands.top; vp < ve; vp++)
    {
      assert ((*vp)->mark_res_cand);
      (*vp)->mark_res_cand = 0;
    }
  QDPLL_RESET_STACK (qdpll->res_cands);

#ifndef NDEBUG
#if QDPLL_ASSERT_LEARN_VARS_UNMARKED
  assert_learn_vars_unmarked (qdpll);
#endif
#endif

  if (success)
    {
      /* Import reason. */
      Constraint *learnt_constraint = 0;
      learnt_constraint =
	create_constraint (qdpll, QDPLL_COUNT_STACK (*lit_stack),
			   type == QDPLL_QTYPE_FORALL);
      assert (type == QDPLL_QTYPE_FORALL || !learnt_constraint->is_cube);
      assert (type == QDPLL_QTYPE_EXISTS || learnt_constraint->is_cube);
      assert (!learnt_constraint->learnt);
      learnt_constraint->learnt = 1;
      p = learnt_constraint->lits;

      unsigned int offset = 0;
      LitID *stack_p, *stack_e;
      for (stack_p = lit_stack->start, stack_e = lit_stack->top;
	   stack_p < stack_e; stack_p++)
	{
	  assert (type == QDPLL_QTYPE_FORALL
		  || p <
		  learnt_constraint->lits + learnt_constraint->num_lits);
	  lit = *stack_p;
	  assert (lit);
	  *p++ = lit;
	  var = LIT2VARPTR (vars, lit);
	  offset++;
	}

      if (type == QDPLL_QTYPE_EXISTS)
	{
	  assert (!learnt_constraint->is_cube);
	  LINK_FIRST (qdpll->pcnf.learnt_clauses, learnt_constraint, link);
	}
      else
	{
	  assert (learnt_constraint->is_cube);
	  LINK_FIRST (qdpll->pcnf.learnt_cubes, learnt_constraint, link);
	}

      assert (QDPLL_COUNT_STACK (*lit_stack) != 0);
      unsigned int asserting_level, max_type_level =
	get_highest_type_lit_dec_level (qdpll, lit_stack->start,
					lit_stack->top, type);
      assert (count_type_lit_at_dec_level
	      (qdpll, lit_stack->start, lit_stack->top, max_type_level,
	       type) == 1);
      Var *type_var =
	get_type_var_at_dec_level (qdpll, lit_stack->start, lit_stack->top,
				   max_type_level, type);

      assert (QDPLL_VAR_ASSIGNED (type_var));

      /* Set forced assignment (by asserting reason) to be enqueued afterwards. */
      assert (type == type_var->scope->type);
      assert (!qdpll->state.forced_assignment.antecedent);
      assert (!qdpll->state.forced_assignment.var);
      assert (!qdpll->state.forced_assignment.assignment);
      assert (!qdpll->state.forced_assignment.mode);
      qdpll->state.forced_assignment.var = type_var;
      qdpll->state.forced_assignment.assignment = -type_var->assignment;
      qdpll->state.forced_assignment.mode = QDPLL_VARMODE_UNIT;

      asserting_level =
	get_reason_asserting_level (qdpll, lit_stack->start, lit_stack->top,
				    type_var, type);

      qdpll->state.forced_assignment.antecedent = learnt_constraint;
      set_learnt_constraint_lit_watchers (qdpll, learnt_constraint,
					  asserting_level);

      if (type == QDPLL_QTYPE_EXISTS)
	{
	  if (qdpll->options.verbosity > 1)
	    {
	      fprintf (stderr, "CDCL: Added learnt clause: ");
	      print_constraint (qdpll, learnt_constraint);
	    }
	}
      else
	{
	  if (qdpll->options.verbosity > 1)
	    {
	      fprintf (stderr, "SDCL: Added learnt cube: ");
	      print_constraint (qdpll, learnt_constraint);
	    }
	}

      decay_var_activity (qdpll);

      if (type == QDPLL_QTYPE_FORALL)
	QDPLL_RESET_STACK (qdpll->sdcl_assigned_vars);
      QDPLL_RESET_STACK (*lit_stack);
      assert (QDPLL_COUNT_STACK (*lit_stack_tmp) == 0);
      return 1 + asserting_level;
    }
  else
    {
      if (type == QDPLL_QTYPE_FORALL)
	QDPLL_RESET_STACK (qdpll->sdcl_assigned_vars);
      QDPLL_RESET_STACK (*lit_stack);
      assert (QDPLL_COUNT_STACK (*lit_stack_tmp) == 0);

      if (qdpll->state.abort_learning)
	{
	  qdpll->state.abort_learning = 0;
	  if (type == QDPLL_QTYPE_FORALL)
	    return analyze_solution_no_sdcl (qdpll);
	  else
	    return analyze_conflict_no_cdcl (qdpll);
	}
      else
	return QDPLL_INVALID_DECISION_LEVEL;
    }
}


static unsigned int
analyze_conflict_cdcl (QDPLL * qdpll)
{
  /* Explicitly handle top-level conflicts. */
  if (qdpll->state.decision_level == 0)
    return QDPLL_INVALID_DECISION_LEVEL;
  else
    return generate_and_add_reason (qdpll, QDPLL_QTYPE_EXISTS);
}


static unsigned int
analyze_conflict (QDPLL * qdpll)
{
  return analyze_conflict_cdcl (qdpll);
}


static unsigned int
analyze_solution_sdcl (QDPLL * qdpll)
{
  /* Explicitly handle top-level conflicts. */
  if (qdpll->state.decision_level == 0)
    return QDPLL_INVALID_DECISION_LEVEL;
  else
    return generate_and_add_reason (qdpll, QDPLL_QTYPE_FORALL);
}


static unsigned int
analyze_solution (QDPLL * qdpll)
{
  return analyze_solution_sdcl (qdpll);
}

/* -------------------- END: LEARNING -------------------- */


static void
backtrack_undo_assignment (QDPLL * qdpll, Var * var, int notify_active)
{
  assert (QDPLL_VAR_ASSIGNED (var));
  assert (var->assignment != QDPLL_ASSIGNMENT_UNDEF);
  assert (var->mode != QDPLL_VARMODE_UNDEF);
  assert (var->decision_level > 0);
  assert (var->decision_level != QDPLL_INVALID_DECISION_LEVEL);
  assert (var->trail_pos != QDPLL_INVALID_TRAIL_POS);
  assert (var->trail_pos <
	  ((unsigned int) (qdpll->assigned_vars_top - qdpll->assigned_vars)));
  assert (qdpll->assigned_vars[var->trail_pos] == var->id);
  QDPLLDepManGeneric *dm = qdpll->dm;

  var->mode = QDPLL_VARMODE_UNDEF;
  var->assignment = QDPLL_ASSIGNMENT_UNDEF;
  var->decision_level = QDPLL_INVALID_DECISION_LEVEL;
  var->trail_pos = QDPLL_INVALID_TRAIL_POS;
  if (var->antecedent)
    {
      assert (var->antecedent->is_reason);
      var->antecedent->is_reason = 0;
      var->antecedent = 0;
    }

  if (var->mark_is_candidate && var->priority_pos == QDPLL_INVALID_PQUEUE_POS)
    var_pqueue_insert (qdpll, var->id, var->priority);

  if (QDPLL_VAR_MARKED_PROPAGATED (var))
    {
      QDPLL_VAR_UNMARK_PROPAGATED (var);
      if (notify_active)
	dm->notify_active (dm, var->id);
    }
}


static void
re_init_deps_at_ignore_level (QDPLL * qdpll)
{
  assert (qdpll->state.decision_level == 0);
  assert (qdpll->bcp_ptr == qdpll->assigned_vars_top);
  assert (qdpll->bcp_ptr == qdpll->old_bcp_ptr);
  QDPLLDepManGeneric *dm = qdpll->dm;
  Var *vars = qdpll->pcnf.vars;
  unsigned int ignore_dlevel = 0;
  dm->reset (dm);
  /* Reset-function has already removed variables from priority
     queue. */
  qdpll->cnt_var_pqueue = 0;
  /* Disable occurrence clauses. */
  assert (qdpll->assigned_vars + qdpll->state.deps_init_trail_disabled <=
	  qdpll->assigned_vars_top);
  VarID *p, *e;
  for (p = qdpll->assigned_vars + qdpll->state.deps_init_trail_disabled,
       e = qdpll->assigned_vars_top; p < e; p++)
    {
      Var *v = VARID2VARPTR (vars, *p);
      if (v->decision_level > ignore_dlevel)
	break;
      else
	{
	  toggle_occurrence_clauses (qdpll, v, 1);
	  qdpll->state.deps_init_trail_disabled++;
	}
    }
  assert (qdpll->assigned_vars + qdpll->state.deps_init_trail_disabled <=
	  qdpll->assigned_vars_top);
  /* Re-initialize dependencies. */
  if (qdpll->options.verbosity > 1)
    fprintf (stderr,
	     "Initializing dependencies at solver level %d, ignore-level %d.\n",
	     qdpll->state.decision_level, 0);
  unsigned int used_vars_before = qdpll->pcnf.used_vars;
  dm->init (dm);
  qdpll->num_deps_init++;
#ifndef NDEBUG
#if QDPLL_ASSERT_RE_INIT_DEPS_DISABLED_CLAUSES
  assert_re_init_deps_disabled_clauses (qdpll);
#endif
#endif

  if (used_vars_before != qdpll->pcnf.used_vars)
    {
      vars = qdpll->pcnf.vars;
      if (qdpll->assigned_vars)
	{
	  VarID *p, *e;
	  for (p = qdpll->assigned_vars_top - 1, e = qdpll->assigned_vars;
	       p >= e; p--)
	    {
	      Var *var = VARID2VARPTR (vars, *p);
	      assert (var->decision_level == 0);
	      assert (QDPLL_VAR_MARKED_PROPAGATED (var));
	      QDPLL_VAR_UNMARK_PROPAGATED (var);
	    }
	}
      qdpll->bcp_ptr = qdpll->old_bcp_ptr = qdpll->assigned_vars;
      reset_watchers (qdpll);
      set_up_watchers (qdpll);
    }
}


/* Undo assignments until 'backtrack_level'. */
static void
backtrack (QDPLL * qdpll, unsigned int backtrack_level)
{
  assert (backtrack_level > 0);
  assert (backtrack_level != QDPLL_INVALID_DECISION_LEVEL);
  assert (QDPLL_ASSIGNMENT_TRUE == -QDPLL_ASSIGNMENT_FALSE);
  assert (QDPLL_ASSIGNMENT_FALSE == -QDPLL_ASSIGNMENT_TRUE);
  assert (qdpll->old_bcp_ptr >= qdpll->assigned_vars);
  assert (qdpll->old_bcp_ptr <= qdpll->bcp_ptr);
  qdpll->state.num_backtracks++;

  VarID *p, *e, *old_bcp_ptr;
  Var *vars = qdpll->pcnf.vars;
  old_bcp_ptr = qdpll->old_bcp_ptr;

  for (p = qdpll->assigned_vars_top - 1, e = qdpll->assigned_vars; p >= e;
       p--)
    {
      Var *assigned_var = VARID2VARPTR (vars, *p);
      assert (QDPLL_VAR_ASSIGNED (assigned_var));
      assert (assigned_var->assignment != QDPLL_ASSIGNMENT_UNDEF);
      assert (assigned_var->decision_level != QDPLL_INVALID_DECISION_LEVEL);
      assert (assigned_var->mode != QDPLL_VARMODE_UNDEF);
      assert (assigned_var->mode != QDPLL_VARMODE_LBRANCH
	      || !assigned_var->antecedent);
      assert (assigned_var->mode != QDPLL_VARMODE_RBRANCH
	      || !assigned_var->antecedent);

      unsigned int var_decision_level = assigned_var->decision_level;
      if (var_decision_level >= backtrack_level)
	backtrack_undo_assignment (qdpll, assigned_var, ((p < old_bcp_ptr)));
      else
	{
	  break;
	}
    }

  qdpll->state.decision_level = backtrack_level - 1;
  assert (qdpll->state.decision_level != QDPLL_INVALID_DECISION_LEVEL);
  qdpll->old_bcp_ptr = qdpll->bcp_ptr = qdpll->assigned_vars_top = p + 1;
}


static Var *
select_decision_variable (QDPLL * qdpll)
{
  QDPLLDepManGeneric *dm = qdpll->dm;
  Var *decision_var, *candidate_var, *vars = qdpll->pcnf.vars;
  VarID candidate, decision_var_id;

  /* Get candidates from dependency manager. */
  while ((candidate = dm->get_candidate (dm)))
    {
      /* Add candidates to priority queue. */
      assert (candidate > 0);
      candidate_var = VARID2VARPTR (vars, candidate);
      assert (dm->is_candidate (dm, candidate));

      if (!QDPLL_VAR_ASSIGNED (candidate_var) &&
	  candidate_var->priority_pos == QDPLL_INVALID_PQUEUE_POS)
	var_pqueue_insert (qdpll, candidate_var->id, candidate_var->priority);
    }

#ifndef NDEBUG
#if QDPLL_ASSERT_CANDIDATES_ON_PQUEUE
  assert_candidates_on_pqueue (qdpll);
#endif
#endif

  do
    {
      decision_var_id = var_pqueue_remove_min (qdpll);
      assert (decision_var_id > 0);
      QDPLL_ABORT_QDPLL (!decision_var_id,
			 "Fatal Error: did not find decision variable!");
      decision_var = VARID2VARPTR (vars, decision_var_id);
      /* Candidates on queue possibly already assigned (unit or pure
         literals). */
      assert (decision_var->priority_pos == QDPLL_INVALID_PQUEUE_POS);
    }
  while (QDPLL_VAR_ASSIGNED (decision_var)
	 || !dm->is_candidate (dm, decision_var_id));

  assert (decision_var->mode == QDPLL_VARMODE_UNDEF);
  assert (!QDPLL_VAR_ASSIGNED (decision_var));
  assert (decision_var->decision_level == QDPLL_INVALID_DECISION_LEVEL);

  return decision_var;
}


static unsigned int
compute_score_from_clause (QDPLL * qdpll, Var * var, Constraint * clause)
{
  assert (!clause->is_cube);
  Var *vars = qdpll->pcnf.vars;
  LitID *p, *e;
  for (p = clause->lits, e = p + clause->num_lits; p < e; p++)
    {
      LitID lit = *p;
      Var *v = LIT2VARPTR (vars, lit);
      if (v != var)
	{
	  if ((QDPLL_LIT_NEG (lit) && QDPLL_VAR_ASSIGNED_FALSE (v)) ||
	      (QDPLL_LIT_POS (lit) && QDPLL_VAR_ASSIGNED_TRUE (v)))
	    return 1;
	}
    }
  return 0;
}


static unsigned int
compute_score (QDPLL * qdpll, Var * var, LitID lit, OccList * occs)
{
  unsigned int sum = 0;
  OccListIterator it;
  OLITER_INIT (it, occs->first, occs->foffset, lit);
  while (OLITER_CUR (it))
    {
      sum += compute_score_from_clause (qdpll, var, OLITER_CUR (it));
      OLITER_NEXT (it, lit);
    }
  return sum;
}


static QDPLLAssignment
dec_heuristic (QDPLL * qdpll, Var * var)
{
  unsigned int pos_score, neg_score;

  neg_score = compute_score (qdpll, var, -var->id, &(var->neg_occ_clauses));
  pos_score = compute_score (qdpll, var, var->id, &(var->pos_occ_clauses));

  if (neg_score < pos_score)
    return QDPLL_ASSIGNMENT_TRUE;
  else
    return QDPLL_ASSIGNMENT_FALSE;
}


static QDPLLAssignment
select_decision_assignment (QDPLL * qdpll, Var * decision_var)
{

  if ((QDPLL_VAR_FORALL (decision_var)) || (QDPLL_VAR_EXISTS (decision_var)))
    {
      assert (QDPLL_ASSIGNMENT_FALSE == -QDPLL_ASSIGNMENT_TRUE);
      /* Return cached assignment if any. */
      QDPLLAssignment a;
      if ((a = decision_var->cached_assignment))
	{
	  if (QDPLL_SCOPE_EXISTS (decision_var->scope))
	    return a;
	  else
	    return a;
	}
    }
  if (QDPLL_VAR_FORALL (decision_var))
    {
      return dec_heuristic (qdpll, decision_var);
    }
  else
    {
      return dec_heuristic (qdpll, decision_var);
    }
}


/* Propagate the effects of setting 'var' to 'true' or 'false'. */
static QDPLLSolverState
propagate_variable_assigned (QDPLL * qdpll, Var * var,
			     LitIDStack * clause_notify_list,
			     ConstraintPtrStack * lit_notify_list)
{
  assert (QDPLL_VAR_ASSIGNED (var));
  assert (var->mode != QDPLL_VARMODE_UNDEF);
  assert (!QDPLL_VAR_MARKED_PROPAGATED (var));
  assert (!QDPLL_VAR_ASSIGNED_TRUE (var)
	  || clause_notify_list == &(var->pos_notify_clause_watchers));
  assert (!QDPLL_VAR_ASSIGNED_TRUE (var)
	  || lit_notify_list == &(var->pos_notify_lit_watchers));
  assert (!QDPLL_VAR_ASSIGNED_FALSE (var)
	  || clause_notify_list == &(var->neg_notify_clause_watchers));
  assert (!QDPLL_VAR_ASSIGNED_FALSE (var)
	  || lit_notify_list == &(var->neg_notify_lit_watchers));
  QDPLLDepManGeneric *dm = qdpll->dm;

  QDPLL_VAR_MARK_PROPAGATED (var);

  /* Notify watching variables. */
  notify_clause_watching_variables (qdpll, clause_notify_list);

  /* Check constraints for units and conflicts/solutions. */
  Constraint **p, **e, *c, *sentinel;
  for (p = lit_notify_list->start, e = lit_notify_list->top; p < e; p++)
    {
      c = *p;
      if (!(sentinel = update_literal_watchers (qdpll, var, c)))
	{
	  if (has_constraint_spurious_pure_lit (qdpll, c))
	    {
	      continue;
	    }

	  assert (c->is_cube || is_clause_empty (qdpll, c));
	  assert (c->is_cube || !is_clause_satisfied (qdpll, c));
	  assert (!c->is_cube || !is_cube_empty (qdpll, c));
	  assert (!c->is_cube || is_cube_satisfied (qdpll, c));
	  assert (!qdpll->result_constraint);

	  if (c->learnt)
	    {
	      learnt_constraint_mtf (qdpll, c);
	      c->res_cnt++;
	    }

	  qdpll->result_constraint = c;
	  if (!c->is_cube)
	    return QDPLL_SOLVER_STATE_UNSAT;
	  else
	    {
	      return QDPLL_SOLVER_STATE_SAT;
	    }
	}
      else if (sentinel != c)
	{
	  /* Sentinel for entry deletion: old last entry has
	     overwritten current one. The constraint's position
	     list has been modified already. */
	  e--;
	  p--;
	}
    }

  /* At this point, state can only be undefined. */
  return QDPLL_SOLVER_STATE_UNDEF;
}


static QDPLLSolverState
bcp (QDPLL * qdpll)
{
  Var *vars = qdpll->pcnf.vars;
  VarID *bcp_ptr;
  QDPLLSolverState state = QDPLL_SOLVER_STATE_UNDEF;

  while (state == QDPLL_SOLVER_STATE_UNDEF &&
	 (bcp_ptr = qdpll->bcp_ptr) < qdpll->assigned_vars_top)
    {
      VarID var_id = *bcp_ptr;
      Var *var = VARID2VARPTR (vars, var_id);
      assert (var->mode != QDPLL_VARMODE_UNDEF);
      assert (QDPLL_VAR_ASSIGNED (var));
      assert (var->decision_level != QDPLL_INVALID_DECISION_LEVEL);
      assert (!QDPLL_VAR_MARKED_PROPAGATED (var));

      if (QDPLL_VAR_ASSIGNED_TRUE (var))
	{
	  state = propagate_variable_assigned (qdpll, var,
					       &
					       (var->
						pos_notify_clause_watchers),
					       &(var->
						 pos_notify_lit_watchers));
	}
      else
	{
	  assert (QDPLL_VAR_ASSIGNED_FALSE (var));
	  state =
	    propagate_variable_assigned (qdpll, var,
					 &(var->neg_notify_clause_watchers),
					 &(var->neg_notify_lit_watchers));
	}

      qdpll->bcp_ptr++;

      if (state == QDPLL_SOLVER_STATE_UNDEF &&
	  qdpll->bcp_ptr == qdpll->assigned_vars_top &&
	  qdpll->pcnf.used_vars ==
	  ((unsigned int) (qdpll->assigned_vars_top - qdpll->assigned_vars)))
	state = QDPLL_SOLVER_STATE_SAT;
    }

#ifndef NDEBUG
#if QDPLL_ASSERT_BCP_WATCHERS_INTEGRITY
  if (state == QDPLL_SOLVER_STATE_UNDEF)
    {
      assert_all_unit_literals_and_literal_watchers_integrity (qdpll);
      assert_all_pure_literals_and_clause_watchers_integrity (qdpll);
    }
#endif
#endif

  return state;
}


static void
notify_inactive_at_decision_point (QDPLL * qdpll)
{
  Var *vars = qdpll->pcnf.vars;
  QDPLLDepManGeneric *dm = qdpll->dm;
  assert (qdpll->bcp_ptr == qdpll->assigned_vars_top);
  assert (qdpll->old_bcp_ptr >= qdpll->assigned_vars);
  assert (qdpll->old_bcp_ptr <= qdpll->bcp_ptr);

  VarID *p, *e;
  for (p = qdpll->old_bcp_ptr, e = qdpll->assigned_vars_top; p < e; p++)
    {
      Var *var = VARID2VARPTR (vars, *p);
      dm->notify_inactive (dm, var->id);
    }

  qdpll->old_bcp_ptr = p;
  assert (qdpll->old_bcp_ptr == qdpll->assigned_vars_top);
}


static void
push_forced_assignment (QDPLL * qdpll)
{
  assert (qdpll->state.forced_assignment.var);
  assert (qdpll->state.forced_assignment.assignment);
  assert (qdpll->state.forced_assignment.mode);

  qdpll->state.forced_assignment.var->antecedent =
    qdpll->state.forced_assignment.antecedent;
  if (qdpll->state.forced_assignment.antecedent)
    {
      assert (!qdpll->state.forced_assignment.antecedent->is_reason);
      qdpll->state.forced_assignment.antecedent->is_reason = 1;
    }

  push_assigned_variable (qdpll, qdpll->state.forced_assignment.var,
			  qdpll->state.forced_assignment.assignment,
			  qdpll->state.forced_assignment.mode);

  qdpll->state.forced_assignment.antecedent = 0;
  qdpll->state.forced_assignment.var = 0;
  qdpll->state.forced_assignment.assignment =
    qdpll->state.forced_assignment.mode = 0;
}


/* Unlink contraint from all lists and release memory. */
static void
cleanup_constraint (QDPLL * qdpll, Constraint * c)
{
  assert (c->learnt);
  assert (!c->is_reason);
  assert (!c->is_watched);
  const int is_cube = c->is_cube;
  /* Unlink constraint from learnt-clause/cube list. */
  if (is_cube)
    UNLINK (qdpll->pcnf.learnt_cubes, c, link);
  else
    UNLINK (qdpll->pcnf.learnt_clauses, c, link);

  /* Delete constraint from lit-watcher notify list. */
  remove_clause_from_notify_list (qdpll,
				  is_cube, 0, c->lits[c->lwatcher_pos], c);
  remove_clause_from_notify_list (qdpll,
				  is_cube, 1, c->lits[c->rwatcher_pos], c);
  delete_constraint (qdpll, c);
}


static void
check_resize_learnt_constraints (QDPLL * qdpll,
				 const QDPLLQuantifierType type)
{
  assert (qdpll->state.lclauses_size);
  assert (qdpll->state.lcubes_size);

  /* Increase constraint sets only if we do not exceed soft space limit. */
  const size_t cur_allocated = qdpll_cur_allocated (qdpll->mm);
  const int cur_exceeded_soft_max_space =
    qdpll->options.soft_max_space &&
    (qdpll->options.soft_max_space * 1024 * 1024 < cur_allocated);
  qdpll->state.exceeded_soft_max_space = qdpll->state.exceeded_soft_max_space
    || cur_exceeded_soft_max_space;

  ConstraintList *constraints;
  Constraint *c;
  if (type == QDPLL_QTYPE_EXISTS)
    {
      if (!cur_exceeded_soft_max_space
	  && qdpll->pcnf.learnt_clauses.cnt < qdpll->state.lclauses_size)
	return;
      constraints = &(qdpll->pcnf.learnt_clauses);
    }
  else
    {
      if (!cur_exceeded_soft_max_space
	  && qdpll->pcnf.learnt_cubes.cnt < qdpll->state.lcubes_size)
	return;
      constraints = &(qdpll->pcnf.learnt_cubes);
    }

  if (type == QDPLL_QTYPE_EXISTS)
    qdpll->state.clause_resizes++;
  else
    qdpll->state.cube_resizes++;

  if (qdpll->options.verbosity > 0)
    fprintf (stderr, "Reduce: %s, cur. size %u, cur cnt %u\n",
	     type == QDPLL_QTYPE_EXISTS ? "clauses" : "cubes",
	     type == QDPLL_QTYPE_EXISTS ? qdpll->state.lclauses_size :
	     qdpll->state.lcubes_size,
	     type ==
	     QDPLL_QTYPE_EXISTS ? qdpll->pcnf.learnt_clauses.cnt : qdpll->
	     pcnf.learnt_cubes.cnt);

  assert (cur_exceeded_soft_max_space || (type == QDPLL_QTYPE_EXISTS &&
					  qdpll->pcnf.learnt_clauses.cnt ==
					  qdpll->state.lclauses_size)
	  || (type == QDPLL_QTYPE_FORALL
	      && qdpll->pcnf.learnt_cubes.cnt == qdpll->state.lcubes_size));

  unsigned int try_delete = type == QDPLL_QTYPE_EXISTS ?
    qdpll->pcnf.learnt_clauses.cnt * (double) 0.5 :
    qdpll->pcnf.learnt_cubes.cnt * (double) 0.5;
  unsigned int del = 0;
  assert (del < try_delete);
  Constraint *prev, *result_constraint = qdpll->result_constraint;
  for (c = constraints->last; c && (del < try_delete); c = prev)
    {
      assert (c->learnt);
      prev = c->link.prev;
      if (!c->is_reason && !c->is_watched && c != result_constraint)
	{
	  cleanup_constraint (qdpll, c);
	  del++;
	}
    }

  assert (type != QDPLL_QTYPE_EXISTS ||
	  qdpll->state.lclauses_size == qdpll->pcnf.learnt_clauses.cnt + del);
  assert (type != QDPLL_QTYPE_FORALL ||
	  qdpll->state.lcubes_size == qdpll->pcnf.learnt_cubes.cnt + del);

  if (!qdpll->state.exceeded_soft_max_space)
    {
      if (type == QDPLL_QTYPE_EXISTS)
	{
	  qdpll->state.lclauses_size += 500;
	}
      else
	{
	  qdpll->state.lcubes_size += 500;
	}
      if (qdpll->options.verbosity > 0)
	fprintf (stderr, "Reduce: del. %d %s, new size %u, new cnt: %u\n",
		 del, type == QDPLL_QTYPE_EXISTS ? "clauses" : "cubes",
		 type ==
		 QDPLL_QTYPE_EXISTS ? qdpll->state.lclauses_size : qdpll->
		 state.lcubes_size,
		 type ==
		 QDPLL_QTYPE_EXISTS ? qdpll->pcnf.learnt_clauses.cnt : qdpll->
		 pcnf.learnt_cubes.cnt);
    }
  else
    {
      if (qdpll->options.verbosity > 0)
	fprintf (stderr,
		 "Reduce: del. %d %s, cur size %u, cur cnt %u, soft limit %u MB reached (alloc.: %f MB)\n",
		 del, type == QDPLL_QTYPE_EXISTS ? "clauses" : "cubes",
		 type ==
		 QDPLL_QTYPE_EXISTS ? qdpll->state.lclauses_size : qdpll->
		 state.lcubes_size,
		 type ==
		 QDPLL_QTYPE_EXISTS ? qdpll->pcnf.learnt_clauses.cnt : qdpll->
		 pcnf.learnt_cubes.cnt, qdpll->options.soft_max_space,
		 cur_allocated / 1024 / (float) 1024);
    }
}


static void
print_config (QDPLL * qdpll)
{
  fprintf (stderr, "\n---------- CONFIG ----------\n");
  fprintf (stderr, "--max-dec=%d\n", qdpll->options.max_dec);
  fprintf (stderr, "--max-space=%d\n", qdpll->options.max_space);
  fprintf (stderr, "--soft-max-space=%d\n", qdpll->options.soft_max_space);

  fprintf (stderr, "----------------------------\n\n");
}


static unsigned int
get_highest_univ_dec_level (QDPLL * qdpll)
{
  Var *var = 0, *vars = qdpll->pcnf.vars;
  VarID *p, *e;
  for (p = qdpll->assigned_vars_top - 1, e = qdpll->assigned_vars; e <= p;
       p--)
    {
      var = VARID2VARPTR (vars, *p);
      if ((var->mode == QDPLL_VARMODE_LBRANCH ||
	   var->mode == QDPLL_VARMODE_RBRANCH)
	  && QDPLL_SCOPE_FORALL (var->scope))
	break;
    }
  /* Must handle pure existential formula. */
  if (!var || var->decision_level == 0)
    return 1;
  else
    return var->decision_level;
}


static int
check_and_restart (QDPLL * qdpll, unsigned int backtrack_level)
{
  if (backtrack_level > 1 && qdpll->state.irestart_dist &&
      (qdpll->state.num_backtracks - qdpll->state.last_backtracks) >=
      qdpll->state.irestart_dist)
    {
      qdpll->state.irestart_dist += 10;
      qdpll->state.num_restarts++;
      qdpll->state.last_backtracks = qdpll->state.num_backtracks;
      qdpll->state.num_irestarts++;
      unsigned int highest_univ = get_highest_univ_dec_level (qdpll);
      unsigned int btlevel = 1;
      btlevel =
	backtrack_level < highest_univ ? backtrack_level : highest_univ;
      backtrack (qdpll, btlevel);
      if (btlevel == backtrack_level)
	push_forced_assignment (qdpll);
      else
	{
	  assert (!qdpll->state.restarting);
	  qdpll->state.restarting = 1;
	  memset (&(qdpll->state.forced_assignment), 0,
		  sizeof (qdpll->state.forced_assignment));
	}
      if (qdpll->options.verbosity > 0)
	fprintf (stderr, "Restart %d, bt %d, inc %d, next dist %d\n",
		 qdpll->state.num_irestarts, qdpll->state.num_backtracks,
		 10, qdpll->state.irestart_dist);

      if (qdpll->state.orestart_dist &&
	  qdpll->state.num_irestarts >= qdpll->state.orestart_dist)
	{
	  qdpll->state.orestart_dist += 5;
	  qdpll->state.irestart_dist = 100;
	  qdpll->state.num_irestarts = 0;
	  qdpll->state.num_restart_resets++;
	  if (qdpll->options.verbosity > 0)
	    fprintf (stderr, "Reset restarts, o-inc %d, next reset %d\n",
		     5, qdpll->state.orestart_dist);
	}
      return 1;
    }
  return 0;
}


/* Solver's core loop. */
static QDPLLResult
solve (QDPLL * qdpll)
{
  assert (!qdpll->state.restarting);
  QDPLLResult result = QDPLL_RESULT_UNKNOWN;
  QDPLLSolverState state = QDPLL_SOLVER_STATE_UNDEF;
  unsigned int backtrack_level;
  Var *decision_var;
  QDPLLAssignment assignment;

  assert (qdpll->num_deps_init == 1);
  qdpll->num_deps_init = 0;

  double lcubes_init_size, lclauses_init_size;

  if (2500 <= qdpll->pcnf.clauses.cnt && qdpll->pcnf.clauses.cnt <= 10000)
    lclauses_init_size = qdpll->pcnf.clauses.cnt;
  else if (qdpll->pcnf.clauses.cnt < 2500)
    lclauses_init_size = 2500;
  else
    lclauses_init_size = 10000;

  if (2500 <= qdpll->pcnf.clauses.cnt && qdpll->pcnf.clauses.cnt <= 10000)
    lcubes_init_size = qdpll->pcnf.clauses.cnt;
  else if (qdpll->pcnf.clauses.cnt < 2500)
    lcubes_init_size = 2500;
  else
    lcubes_init_size = 10000;

  qdpll->state.lclauses_size = lclauses_init_size;
  qdpll->state.lcubes_size = lcubes_init_size;

  while (1)
    {
      state = bcp (qdpll);

      if (state == QDPLL_SOLVER_STATE_UNSAT)
	{
	  /* Conflict: analyze conflict and backtrack. */
	  assert (qdpll->result_constraint
		  && !qdpll->result_constraint->is_cube);
	  assert (is_clause_empty (qdpll, qdpll->result_constraint));
	  assert (!is_clause_satisfied (qdpll, qdpll->result_constraint));

	  check_resize_learnt_constraints (qdpll, QDPLL_QTYPE_EXISTS);

#if QDPLL_ASSERT_SOLVE_STATE
	  assert (is_formula_false (qdpll));
	  assert (!is_formula_true (qdpll));
#endif
	  backtrack_level = analyze_conflict (qdpll);
	  if (backtrack_level == QDPLL_INVALID_DECISION_LEVEL)
	    {
	      /* Conflict can not be resolved -> terminate. */
	      result = QDPLL_RESULT_UNSAT;
	      break;
	    }
	  else
	    {
	      if (!check_and_restart (qdpll, backtrack_level))
		{
		  backtrack (qdpll, backtrack_level);
		  push_forced_assignment (qdpll);
		}
	    }

	  /* Conflict must be fixed now. */
	  assert (!is_clause_empty (qdpll, qdpll->result_constraint));
	  qdpll->result_constraint = 0;
	}
      else if (state == QDPLL_SOLVER_STATE_SAT)
	{
	  assert (!qdpll->result_constraint
		  || qdpll->result_constraint->is_cube);
	  assert (!qdpll->result_constraint
		  || is_cube_satisfied (qdpll, qdpll->result_constraint));
	  assert (!qdpll->result_constraint
		  || !is_cube_empty (qdpll, qdpll->result_constraint));

	  check_resize_learnt_constraints (qdpll, QDPLL_QTYPE_FORALL);

	  /* Empty formula: analyze solution and backtrack. */
#if QDPLL_ASSERT_SOLVE_STATE
	  assert (!is_formula_false (qdpll));
	  assert (is_formula_true (qdpll));
#endif
	  backtrack_level = analyze_solution (qdpll);
	  if (backtrack_level == QDPLL_INVALID_DECISION_LEVEL)
	    {
	      /* All branches satisfied -> terminate. */
	      result = QDPLL_RESULT_SAT;
	      break;
	    }
	  else
	    {
	      if (!check_and_restart (qdpll, backtrack_level))
		{
		  backtrack (qdpll, backtrack_level);
		  push_forced_assignment (qdpll);
		}
	    }

	  /* Solution must be broken now. */
	  assert (!qdpll->result_constraint
		  || !is_cube_satisfied (qdpll, qdpll->result_constraint));
	  qdpll->result_constraint = 0;
	}
      else
	{
	  /* Result undefined: decide next branch. */
	  if (qdpll->options.max_dec)
	    {
	      qdpll->state.num_decisions++;
	      if (qdpll->options.max_dec < qdpll->state.num_decisions)
		{
		  if (qdpll->options.verbosity > 1)
		    fprintf (stderr, "Aborting after decision limit of %d.\n",
			     qdpll->options.max_dec);
		  return QDPLL_RESULT_UNKNOWN;
		}
	    }

#if QDPLL_ASSERT_SOLVE_STATE
	  assert (!is_formula_false (qdpll));
#endif
	  assert (state == QDPLL_SOLVER_STATE_UNDEF);
	  assert (qdpll->bcp_ptr == qdpll->assigned_vars_top);

	  if (qdpll->state.decision_level > 0)
	    notify_inactive_at_decision_point (qdpll);
	  else
	    {
	      /* Must set propagation pointers. This would normally
	         have been done inside
	         'notify_inactive_at_decision_point'. */
	      assert (qdpll->state.decision_level <= 0);
	      qdpll->old_bcp_ptr = qdpll->assigned_vars_top;
	    }

	  if (!qdpll->state.restarting)
	    {
	      if (qdpll->state.decision_level == 0)
		qdpll->state.pending_inits = 1;
	      if (qdpll->state.pending_inits)
		{
		  assert (qdpll->bcp_ptr == qdpll->assigned_vars_top);
		  re_init_deps_at_ignore_level (qdpll);
		  qdpll->state.pending_inits = 0;
		}
	    }
	  else
	    qdpll->state.restarting = 0;

	  if (qdpll->bcp_ptr != qdpll->assigned_vars_top)
	    continue;

	  decision_var = select_decision_variable (qdpll);
	  assignment = select_decision_assignment (qdpll, decision_var);
	  push_assigned_variable (qdpll, decision_var, assignment,
				  QDPLL_VARMODE_LBRANCH);
	}
    }

  return result;
}


static int
isnumstr (char *str)
{
  /* Empty string is not considered as number-string. */
  if (!*str)
    return 0;
  char *p;
  for (p = str; *p; p++)
    {
      char c = *p;
      if (c != '.' && !isdigit (c))
	return 0;
    }
  return 1;
}


/* -------------------- START: PUBLIC FUNCTIONS --------------------*/


QDPLL *
qdpll_create ()
{
  QDPLLMemMan *mm = qdpll_create_mem_man ();
  QDPLL *qdpll = (QDPLL *) qdpll_malloc (mm, sizeof (QDPLL));
  qdpll->mm = mm;
  Scope *default_scope = (Scope *) qdpll_malloc (mm, sizeof (Scope));
  default_scope->type = QDPLL_QTYPE_EXISTS;
  assert (default_scope->nesting == QDPLL_DEFAULT_SCOPE_NESTING);
  LINK_LAST (qdpll->pcnf.scopes, default_scope, link);
  qdpll->pcnf.size_vars = DEFAULT_VARS_SIZE;
  qdpll->pcnf.vars =
    (Var *) qdpll_malloc (mm, DEFAULT_VARS_SIZE * sizeof (Var));

  /* Set default options. */
  qdpll->dm =
    (QDPLLDepManGeneric *) qdpll_qdag_dep_man_create (qdpll->mm,
						      &(qdpll->pcnf), qdpll);

  qdpll->state.var_act_inc = 1.0;
  qdpll->state.irestart_dist = 100;
  qdpll->state.orestart_dist = 10;

  qdpll->num_deps_init = 1;
  return qdpll;
}


void
qdpll_delete (QDPLL * qdpll)
{
  QDPLL_ABORT_QDPLL (!qdpll, "'qdpll' is null!");
  QDPLLMemMan *mm = qdpll->mm;

  QDPLL_DELETE_STACK (mm, qdpll->add_stack);
  QDPLL_DELETE_STACK (mm, qdpll->add_stack_tmp);
  QDPLL_DELETE_STACK (mm, qdpll->wreason_a);
  QDPLL_DELETE_STACK (mm, qdpll->wreason_e);
  QDPLL_DELETE_STACK (mm, qdpll->choose_var_aclasses);
  QDPLL_DELETE_STACK (mm, qdpll->choose_var_eclasses);
  QDPLL_DELETE_STACK (mm, qdpll->res_cands);
  QDPLL_DELETE_STACK (mm, qdpll->sdcl_assigned_vars);
  /* Delete scopes. */
  Scope *s, *n;
  for (s = qdpll->pcnf.scopes.first; s; s = n)
    {
      n = s->link.next;
      delete_scope (qdpll, s);
    }

  /* Delete variables. */
  Var *vars = qdpll->pcnf.vars;
  Var *v, *ve;
  for (v = vars, ve = vars + qdpll->pcnf.size_vars; v < ve; v++)
    {
      if (v->id)
	delete_variable (qdpll, v);
    }
  qdpll_free (mm, vars, qdpll->pcnf.size_vars * sizeof (Var));

  /* Delete clauses. */
  ConstraintList *constraints = &(qdpll->pcnf.clauses);
  Constraint *c, *nc;
  for (c = constraints->first; c; c = nc)
    {
      nc = c->link.next;
      assert (!c->is_cube);
      assert (!c->learnt);
      delete_constraint (qdpll, c);
    }

  /* Delete learnt clauses. */
  constraints = &(qdpll->pcnf.learnt_clauses);
  for (c = constraints->first; c; c = nc)
    {
      nc = c->link.next;
      assert (!c->is_cube);
      assert (c->learnt);
      delete_constraint (qdpll, c);
    }

  /* Delete learnt cubes. */
  constraints = &(qdpll->pcnf.learnt_cubes);
  for (c = constraints->first; c; c = nc)
    {
      nc = c->link.next;
      assert (c->is_cube);
      assert (c->learnt);
      delete_constraint (qdpll, c);
    }

  qdpll_free (mm, qdpll->var_pqueue, qdpll->size_var_pqueue * sizeof (VarID));

  qdpll_free (mm, qdpll->assigned_vars,
	      size_assigned_vars (qdpll) * sizeof (VarID));

  assert (qdpll->dm);
  /* Delete dependency manager. */
  qdpll_qdag_dep_man_delete ((QDPLLDepManQDAG *) qdpll->dm);

  qdpll_free (mm, qdpll, sizeof (QDPLL));
  qdpll_delete_mem_man (mm);
}


/* Configure solver instance via configuration string. 
   Returns null pointer on success and error string otherwise. */
char *
qdpll_configure (QDPLL * qdpll, char *configure_str)
{
  char *result = 0;

  if (!strncmp (configure_str, "--max-space=", strlen ("--max-space=")))
    {
      configure_str += strlen ("--max-space=");
      if (isnumstr (configure_str))
	{
	  qdpll->options.max_space = atoi (configure_str);
	  /* Space limit takes effect immediately when set. */
	  qdpll_set_mem_limit (qdpll->mm, qdpll->options.max_space);
	}
      else
	result = "Expecting number after '--max-space='";
    }
  else
    if (!strncmp
	(configure_str, "--soft-max-space=", strlen ("--soft-max-space=")))
    {
      configure_str += strlen ("--soft-max-space=");
      if (isnumstr (configure_str))
	{
	  qdpll->options.soft_max_space = atoi (configure_str);
	}
      else
	result = "Expecting number after '--soft-max-space='";
    }
  else if (!strncmp (configure_str, "--max-dec=", strlen ("--max-dec=")))
    {
      configure_str += strlen ("--max-dec=");
      if (isnumstr (configure_str))
	qdpll->options.max_dec = atoi (configure_str);
      else
	result = "Expecting number after '--max-dec='";
    }
  else if (!strcmp (configure_str, "-v"))
    {
      qdpll->options.verbosity++;
    }
  else
    {
      result = "unknown option";
    }

  return result;
}


void
qdpll_adjust_vars (QDPLL * qdpll, VarID num)
{
  QDPLL_ABORT_QDPLL (!qdpll, "'qdpll' is null!");
  QDPLL_ABORT_QDPLL (num == 0, "'num' must not be zero!");
  VarID size_vars = qdpll->pcnf.size_vars;
  /* Index 0 is never used in variable table, hence increase 'num' */
  if (size_vars < ++num)
    {
      qdpll->pcnf.vars = (Var *) qdpll_realloc (qdpll->mm, qdpll->pcnf.vars,
						size_vars * sizeof (Var),
						num * sizeof (Var));
      qdpll->pcnf.size_vars = num;
    }
}


unsigned int
qdpll_new_scope (QDPLL * qdpll, QDPLLQuantifierType qtype)
{
  QDPLL_ABORT_QDPLL (!qdpll, "'qdpll' is null!");
  QDPLL_ABORT_QDPLL ((qtype != QDPLL_QTYPE_EXISTS &&
		      qtype != QDPLL_QTYPE_FORALL), "invalid 'qtype'!");
  QDPLL_ABORT_QDPLL (qdpll->state.scope_opened,
		     "there is already a new, open scope!");
  /* There must be at least a default scope. */
  assert (qdpll->pcnf.scopes.first);
  assert (qdpll->pcnf.scopes.last);
  assert (qdpll->pcnf.scopes.first != qdpll->pcnf.scopes.last ||
	  (QDPLL_SCOPE_EXISTS (qdpll->pcnf.scopes.first) &&
	   qdpll->pcnf.scopes.first->nesting == QDPLL_DEFAULT_SCOPE_NESTING));

  qdpll->state.scope_opened = 1;
  unsigned int nesting = qdpll->pcnf.scopes.last->nesting + 1;
  assert (nesting > 0);
  Scope *scope = (Scope *) qdpll_malloc (qdpll->mm, sizeof (Scope));
  scope->nesting = nesting;
  scope->type = qtype;
  LINK_LAST (qdpll->pcnf.scopes, scope, link);
  return nesting;
}


void
qdpll_add (QDPLL * qdpll, LitID id)
{
  QDPLL_ABORT_QDPLL (!qdpll, "'qdpll' is null!");
  const char *err_msg;

  QDPLLMemMan *mm = qdpll->mm;
  LitIDStack *add_stack = &(qdpll->add_stack);

  if (id == 0)
    {
      /* '0' closes a scope or clause */
      err_msg = import_added_ids (qdpll);
      QDPLL_ABORT_QDPLL (err_msg, err_msg);
      assert (!qdpll->state.scope_opened);
    }
  else
    QDPLL_PUSH_STACK (mm, *add_stack, id);
}


QDPLLResult
qdpll_sat (QDPLL * qdpll)
{
  if (qdpll->options.verbosity > 0)
    print_config (qdpll);

  QDPLLMemMan *mm = qdpll->mm;
  QDPLLDepManGeneric *dm = qdpll->dm;
  assert (dm);
  QDPLLResult r = QDPLL_RESULT_UNKNOWN;

  if (qdpll->state.found_empty_clause)
    r = QDPLL_RESULT_UNSAT;
  else if (!qdpll->pcnf.clauses.first)
    {
      /* Empty formula. */
      assert (!qdpll->pcnf.clauses.last);
      r = QDPLL_RESULT_SAT;
    }
  else
    {
      /* Decide formula. */
      r = set_up_formula (qdpll);
      if (r == QDPLL_SOLVER_STATE_UNDEF)
	{
#ifndef NDEBUG
#if QDPLL_ASSERT_FULL_FORMULA_INTEGRITY
	  assert_full_formula_integrity (qdpll);
#endif
#endif
	  r = solve (qdpll);
	}
    }

  return r;
}


void
qdpll_print (QDPLL * qdpll, FILE * out)
{
  if (qdpll->state.found_empty_clause)
    {
      /* Print out the empty clause and return. All other clauses are
         ignored. */
      fprintf (out, "p cnf %d %d\n", 1, 1);
      fprintf (out, "0\n");
      return;
    }

  clean_up_formula (qdpll);

#ifndef NDEBUG
#if QDPLL_ASSERT_FULL_FORMULA_INTEGRITY
  assert_full_formula_integrity (qdpll);
#endif
#endif
  assert (qdpll->pcnf.clauses.cnt ==
	  count_constraints (&(qdpll->pcnf.clauses)));

  fprintf (out, "p cnf %d %d\n", qdpll->pcnf.max_declared_var_id,
	   qdpll->pcnf.clauses.cnt);

  assert (qdpll->pcnf.scopes.first);
  assert (qdpll->pcnf.scopes.first->nesting == QDPLL_DEFAULT_SCOPE_NESTING);
  assert (QDPLL_SCOPE_EXISTS (qdpll->pcnf.scopes.first));

  Scope *s;
  for (s = qdpll->pcnf.scopes.first; s; s = s->link.next)
    {
      if (QDPLL_COUNT_STACK (s->vars) == 0)
	continue;

      if (QDPLL_SCOPE_EXISTS (s))
	fprintf (out, "e");
      else
	fprintf (out, "a");

      VarID *p, *e;
      for (p = s->vars.start, e = s->vars.top; p < e; p++)
	fprintf (out, " %u", *p);
      fprintf (out, " 0\n");
    }

  Constraint *c;
  for (c = qdpll->pcnf.clauses.first; c; c = c->link.next)
    {
      assert (!c->is_cube);
      assert (c->num_lits > 0);
      int *p = c->lits, *e;
      fprintf (out, "%d", *p);
      for (p++, e = c->lits + c->num_lits; p < e; p++)
	fprintf (out, " %d", *p);
      fprintf (out, " 0\n");
    }
}


/* -------------------- END: PUBLIC FUNCTIONS --------------------*/
