/*
 This file is part of DepQBF.

 DepQBF, a solver for quantified boolean formulae (QBF).	
 Copyright 2010 Florian Lonsing, Johannes Kepler University, Linz, Austria.

 See also http://fmv.jku.at/depqbf for further information.

 DepQBF is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or (at
 your option) any later version.

 DepQBF is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with DepQBF.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <limits.h>
#include <dirent.h>
#include <assert.h>
#include <string.h>
#include <stdarg.h>
#include <signal.h>
#include <unistd.h>
#include "qdpll_exit.h"
#include "qdpll_config.h"
#include "qdpll.h"

#define QDPLL_ABORT_APP(cond,msg) \
  do {				  \
    if (cond)								\
      {									\
        fprintf (stderr, "[QDPLL-APP] %s: %s\n", __func__, msg);	\
	fflush (stderr);						\
        abort();							\
      }									\
  } while (0)


struct QDPLLApp
{
  struct
  {
    char *in_filename;
    FILE *in;
    int pretty_print;
    unsigned int max_time;
    unsigned int verbosity;
  } options;
};

typedef struct QDPLLApp QDPLLApp;

static void
print_abort_err (QDPLLApp * app, char *msg, ...)
{
  va_list list;
  assert (msg != NULL);
  fprintf (stderr, "qdpll-app: ");
  va_start (list, msg);
  vfprintf (stderr, msg, list);
  va_end (list);
  fflush (stderr);
  abort ();
}

#define VERSION								\
  "DepQBF 0.1\n"							\
  "Copyright 2010 Florian Lonsing, Johannes Kepler University, Linz, Austria.\n"\
  "This is free software; see COPYING for copying conditions.\n"	\
  "There is NO WARRANTY, to the extent permitted by law.\n"


#define USAGE								\
  "depqbf [<options>] [<file>]\n"						\
  "\n"									\
  "  where <file> is a QBF in QDIMACS format (default: stdin)\n"	\
  "\n"									\
  "  where <options> is a combination of:\n"				\
  "            -h, --help:   print usage information.\n"		\
  "             --version:   print version.\n"				\
  "        --pretty-print:   only parse and print formula.\n"		\
  "                    -v:   increase verbosity incrementally.\n"	\
  "                   val:   time-out after 'val' seconds.\n\n"		\
  "  Experimental options:\n"						\
  "       --max-space=val:   abort if more than 'val' MB allocated.\n"	\
  "  --soft-max-space=val:   try to allocate max. 'val' MB, but do not abort.\n" \
  "         --max-dec=val:   abort after 'val' decisions made.\n"	\
  "\n"									\
  "  Exit codes:\n"							\
  "    10 if <file> is satisfiable.\n"					\
  "    20 if <file> is unsatisfiable.\n"				\
  "    Any other exit code indicates that <file> was not solved.\n"	\
  "\n"									\
  "  Project webpage: http://fmv.jku.at/depqbf\n\n"

/* -------------------- START: PARSING -------------------- */
#define PARSER_READ_NUM(num, c)			\
  assert (isdigit (c));				\
  num = 0;					\
  do						\
    {						\
      num = num * 10 + (c - '0');		\
    }						\
  while (isdigit ((c = getc (in))));

#define PARSER_SKIP_SPACE_DO_WHILE(c)		\
  do						\
    {						\
      c = getc (in);				\
    }						\
  while (isspace (c));

#define PARSER_SKIP_SPACE_WHILE(c)		\
  while (isspace (c))				\
    c = getc (in);


static void
parse (QDPLLApp * app, QDPLL * qdpll, FILE * in)
{
  int col = 0, line = 0, neg = 0, preamble_found = 0;
  LitID num = 0;
  QDPLLQuantifierType scope_type = QDPLL_QTYPE_UNDEF;

  assert (in);

  char c;
  while ((c = getc (in)) != EOF)
    {
      PARSER_SKIP_SPACE_WHILE (c);

      while (c == 'c')
	{
	  while ((c = getc (in)) != '\n' && c != EOF)
	    ;
	  c = getc (in);
	}

      PARSER_SKIP_SPACE_WHILE (c);

      if (c == 'p')
	{
	  /* parse preamble */
	  PARSER_SKIP_SPACE_DO_WHILE (c);
	  if (c != 'c')
	    goto MALFORMED_PREAMBLE;
	  PARSER_SKIP_SPACE_DO_WHILE (c);
	  if (c != 'n')
	    goto MALFORMED_PREAMBLE;
	  PARSER_SKIP_SPACE_DO_WHILE (c);
	  if (c != 'f')
	    goto MALFORMED_PREAMBLE;
	  PARSER_SKIP_SPACE_DO_WHILE (c);
	  if (!isdigit (c))
	    goto MALFORMED_PREAMBLE;

	  /* read number of variables */
	  PARSER_READ_NUM (num, c);

	  PARSER_SKIP_SPACE_WHILE (c);
	  if (!isdigit (c))
	    goto MALFORMED_PREAMBLE;

	  /* read number of clauses */
	  PARSER_READ_NUM (num, c);

	  preamble_found = 1;
	  goto PARSE_SCOPE_OR_CLAUSE;

	MALFORMED_PREAMBLE:
	  QDPLL_ABORT_APP (1, "malformed preamble!\n");
	  return;
	}
      else
	{
	  QDPLL_ABORT_APP (1, "expecting preamble!\n");
	  return;
	}

    PARSE_SCOPE_OR_CLAUSE:

      PARSER_SKIP_SPACE_WHILE (c);

      if (c == 'a' || c == 'e')
	{
	  /* open a new scope */
	  if (c == 'a')
	    scope_type = QDPLL_QTYPE_FORALL;
	  else
	    scope_type = QDPLL_QTYPE_EXISTS;

	  qdpll_new_scope (qdpll, scope_type);

	  PARSER_SKIP_SPACE_DO_WHILE (c);
	}

      if (!isdigit (c) && c != '-')
	{
	  if (c == EOF)
	    return;
	  QDPLL_ABORT_APP (1, "expecting digit or '-'!\n");
	  return;
	}
      else
	{
	  if (c == '-')
	    {
	      neg = 1;
	      if (!isdigit ((c = getc (in))))
		{
		  QDPLL_ABORT_APP (1, "expecting digit!\n");
		  return;
		}
	    }

	  /* parse a literal or variable ID */
	  PARSER_READ_NUM (num, c);
	  num = neg ? -num : num;
	  qdpll_add (qdpll, num);
	  neg = 0;

	  goto PARSE_SCOPE_OR_CLAUSE;
	}
    }

  if (!preamble_found)
    QDPLL_ABORT_APP (1, "preamble missing!\n");
}

/* -------------------- END: PARSING -------------------- */



static void
check_options (QDPLLApp * app)
{

}


static void
set_default_options (QDPLLApp * app)
{
  app->options.in_filename = 0;
  app->options.in = stdin;
  app->options.pretty_print = 0;
}


static int
isnumstr (char *str)
{
  /* Empty string is not considered as number-string. */
  if (!*str)
    return 0;
  char *p;
  for (p = str; *p; p++)
    {
      if (!isdigit (*p))
	return 0;
    }
  return 1;
}


static int
parse_cmd_line_options (QDPLLApp * app, QDPLL * qdpll, int argc, char **argv)
{
  char *result;
  int done = 0;
  int opt_cnt;
  for (opt_cnt = 1; opt_cnt < argc; opt_cnt++)
    {
      char *opt_str = argv[opt_cnt];

      if (!strcmp (opt_str, "-h") || !strcmp (opt_str, "--help"))
	{
	  fprintf (stderr, "%s", USAGE);
	  return 1;
	}
      else if (!strcmp (opt_str, "--pretty-print"))
	{
	  app->options.pretty_print = 1;
	}
      else if (!strcmp (opt_str, "--version"))
	{
	  fprintf (stderr, "%s\n", VERSION);
	  return 1;
	}
      else if (isnumstr (opt_str))
	{
	  app->options.max_time = atoi (opt_str);
	  if (app->options.max_time == 0)
	    {
	      result = "Expecting non-zero value for max-time";
	      print_abort_err (app, "%s!\n\n", result);
	    }
	}
      else if (!strncmp (opt_str, "-", 1) || !strncmp (opt_str, "--", 2))
	{
	  /* Handle QDPLL-options.  TODO: options unknown to QDPLL
	     should return specific error-message. Now only 'unknown
	     option' is returned. */
	  if (!strcmp (opt_str, "-v"))
	    {
	      app->options.verbosity++;
	    }
	  if ((result = qdpll_configure (qdpll, opt_str)))
	    print_abort_err (app, "%s!\n\n", result);
	}
      else if (!app->options.in_filename)
	{
	  app->options.in_filename = opt_str;
	  /* Check input file. */
	  DIR *dir;
	  if ((dir = opendir (app->options.in_filename)) != NULL)
	    {
	      closedir (dir);
	      print_abort_err (app, "input file '%s' is a directory!\n\n",
			       app->options.in_filename);
	    }
	  FILE *input_file = fopen (app->options.in_filename, "r");
	  if (!input_file)
	    {
	      print_abort_err (app, "could not open input file '%s'!\n\n",
			       app->options.in_filename);
	    }
	  else
	    app->options.in = input_file;
	}
      else
	{
	  print_abort_err (app, "unknown option '%s'!\n\n", opt_str);
	}
    }
  return done;
}


static void
sig_handler (int sig)
{
  fprintf (stderr, "\n\n SIG RECEIVED\n\n");
  raise (sig);
}


static void
sigalrm_handler (int sig)
{
  fprintf (stderr, "\n\n SIGALRM RECEIVED\n\n");
  raise (sig);
}


static void
set_signal_handlers (QDPLLApp * app)
{
  signal (SIGINT, sig_handler);
  signal (SIGTERM, sig_handler);
  signal (SIGALRM, sigalrm_handler);
}


int
qdpll_main (int argc, char **argv)
{
  QDPLLResult result = QDPLL_RESULT_UNKNOWN;
  QDPLLApp app;
  memset (&app, 0, sizeof (QDPLLApp));
  set_default_options (&app);

  QDPLL *qdpll = qdpll_create ();

  if (!parse_cmd_line_options (&app, qdpll, argc, argv))
    {
      check_options (&app);
      set_signal_handlers (&app);

      if (app.options.max_time)
	alarm (app.options.max_time);

      parse (&app, qdpll, app.options.in);

      if (app.options.pretty_print)
	qdpll_print (qdpll, stdout);
      else
	{
	  result = qdpll_sat (qdpll);
	}

      if (result == QDPLL_RESULT_SAT)
	fprintf (stdout, "SAT\n");
      else if (result == QDPLL_RESULT_UNSAT)
	fprintf (stdout, "UNSAT\n");
      else if (!app.options.pretty_print)
	{
	  assert (result == QDPLL_RESULT_UNKNOWN);
	  fprintf (stdout, "UNKNOWN\n");
	}
    }
  qdpll_delete (qdpll);

  if (app.options.in_filename)
    fclose (app.options.in);

  return result;
}
