/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#ifndef TIMELINE_H
#define TIMELINE_H

#include <QWidget>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QVariantList>
#include <QMap>
#include <QPushButton>
#include <QtCore/qmath.h>

#include <QDebug>

#include "pumpcontroller.h"
#include "globalobject.h"
#include "asobject.h"
#include "asactivity.h"
#include "post.h"
#include "filterchecker.h"


class TimeLine : public QWidget
{
    Q_OBJECT

public:
    TimeLine(int timelineType,
             PumpController *pumpController,
             GlobalObject *globalObject,
             FilterChecker *filterChecker,
             QWidget *parent = 0);
    ~TimeLine();

    void clearTimeLineContents();
    void requestTimelinePage();
    void updateCurrentPageNumber();

    void setMinMaxHeightForPosts(int newMinMaxHeightForPosts);
    void resizePosts();

    void markPostsAsRead();

    void updateFuzzyTimestamps();

    void updateHighlightColors(QStringList colorList);



    enum TimelineTypes
    {
        TimelineTypeMain,
        TimelineTypeDirect,
        TimelineTypeActivity,
        TimelineTypeFavorites
    };



signals:
    void getPreviousPage();
    void getNextPage();

    void scrollTo(QAbstractSlider::SliderAction sliderAction);

    void timelineRendered(int timelineType,
                          int newPostCount,
                          int highlightedPostsCount,
                          int totalItemsCount);
    void unreadPostsCountChanged(int timelineType,
                                 int newPostCount,
                                 int highlightedCount,
                                 int totalItemsCount);

    void commentingOnPost(QWidget *commenterWidget);


public slots:
    void setTimeLineContents(QVariantList postList, int postsPerPage,
                             QString previousLink, QString nextLink,
                             int totalItems);

    void setLikesInPost(QVariantList likesList, QString originatingPostURL);
    void setCommentsInPost(QVariantList commentsList, QString originatingPostURL);

    void goToFirstPage();
    void goToPreviousPage();
    void goToNextPage();

    void scrollUp();
    void scrollDown();
    void scrollPageUp();
    void scrollPageDown();
    void scrollToTop();
    void scrollToBottom();

    void decreaseUnreadPostsCount(bool wasHighlighted);

protected:


private:
    QVBoxLayout *mainLayout;
    QVBoxLayout *postsLayout;
    QHBoxLayout *bottomLayout;

    PumpController *pController;
    GlobalObject *globalObj;
    FilterChecker *fChecker;

    QPushButton *firstPageButton;
    QLabel *currentPageLabel;
    QPushButton *previousPageButton;
    QPushButton *nextPageButton;

    QString previousPageLink;
    QString nextPageLink;

    QAction *scrollUpAction;
    QAction *scrollDownAction;
    QAction *scrollPageUpAction;
    QAction *scrollPageDownAction;
    QAction *scrollTopAction;
    QAction *scrollBottomAction;

    QAction *previousPageAction;
    QAction *nextPageAction;

    int timelineType;

    int timelineOffset;
    int postsPerPage;
    int unreadPostsCount;
    int newHighlightedPostsCount;
    int totalItemsCount;

    int minMaxHeightForPosts;

    QStringList highlightColorsList;

    QString previousNewestPostId;

    QList<Post *> postsInTimeline;
};

#endif // TIMELINE_H
