/*
 * @(#)MinDirChannelLogger.java
 *
 * Copyright (C) 2003-2004 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.logger;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;


/**
 * Same as the DirectoryChannelLogger, but includes a JDK 1.2 optimization
 * to minimize the size of the log files.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @author    Graf Leonardo <a href="mailto:leolegenie@hotmail.com">leolegenie@hotmail.com</a>
 * @version   $Date: 2004/07/07 09:39:10 $
 * @since     December 3, 2003
 */
public class MinDirChannelLogger extends DirectoryChannelLogger
{
    private final Set covered = new HashSet( 10000, 0.75f );
    
    
    public MinDirChannelLogger( File baseDir )
    {
        super( baseDir );
    }
    
    
    /**
     * Records a coverage of a marked bytecode instruction.  This method should
     * never throw an exception.
     *
     * @param classSignature a signature of the class file being covered.
     *        this signature includes the fully-qualified name of the class,
     *        along with a checksum to uniquely identify it.
     * @param methodIndex index for a method within the class.  The meta-data
     *        store will know how to translate the index to a method signature.
     * @param markIndex the index of the bytecode instruction mark for this
     *        particular channel.
     */
    public void cover( String classSignature, short methodIndex,
            short markIndex )
    {
        if (this.baseDir != null)
        {
            char[] out = createCoverString( methodIndex, markIndex );
            
            // optimize
            // String key = classSignature + '.' + out;
            int csl = classSignature.length();
            char cs[] = new char[ csl + 3 ];
            classSignature.getChars( 0, csl, cs, 0 );
            cs[csl] = '-';
            cs[csl+1] = (char)methodIndex;
            cs[csl+2] = (char)markIndex;
            String key = new String( cs );
            
            synchronized (this)
            {
                // access to covered needs to be inside the synchronized block
                if (!covered.contains( key ))
                {
                    covered.add( key );
                    
                    File f = getClassFile( this.baseDir, classSignature );
                    FileWriter fw = null;
                    try
                    {
                        fw = new FileWriter( f.toString(), true );
                        try
                        {
                            fw.write( out );
                            fw.flush();
                        }
                        finally
                        {
                            fw.close();
                        }
                    }
                    catch (IOException ioe)
                    {
                        // don't throw this exception outside this block.
                        ioe.printStackTrace();
                        
                        // prevent these errors from occuring again
                        this.baseDir = null;
                    }
                }
            }
        }
    }
}

