/*
 * @(#)SysPropertiesUtilUTest.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.junit.v1;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import java.util.Properties;
import java.util.Enumeration;

/**
 * Tests the TestRunnable class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since     March 1, 2002
 * @version   $Date: 2003/02/10 22:52:23 $
 */
public class SysPropertiesUtilUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = SysPropertiesUtilUTest.class;
    
    public SysPropertiesUtilUTest( String name )
    {
        super( name );
    }

    


    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testConstructor1()
    {
        new SysPropertiesUtil();
    }
    public void testSetValue1()
    {
        String uk = getUniqueKey();
        spu.setValue( uk, "b" );
        assertEquals(
            "Did not set unique value correctly.",
            "b",
            getValue( uk ) );
    }
    
    
    public void testSetValue2()
    {
        try
        {
            spu.setValue( null, "a" );
            fail(
                "Did not throw IllegalArgumentException for null value." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception
        }
    }
    
    
    public void testSetValue3()
    {
        String uk = getUniqueKey();
        spu.setValue( uk, null );
        assertNull(
            "Did not set/do nothing with unique value.",
            getValue( uk ) );
    }
    
    
    public void testSetValue4()
    {
        String ek = getExistingKey();
        String val = getValue( ek );
        assertNotNull(
            "The existing value in the set should NOT be null!",
            val );
        String newVal = val + "-not real";
        spu.setValue( ek, newVal );
        assertEquals(
            "Did not set existing value correctly.",
            newVal,
            getValue( ek ) );
    }
    
    
    public void testSetValue5()
    {
        String ek = getExistingKey();
        spu.setValue( ek, null );
        assertNull(
            "Did not remove key with existing value.",
            getValue( ek ) );
    }
    
    
    public void testReset1()
    {
        String ek = getExistingKey();
        String val = getValue( ek );
        assertNotNull(
            "The existing value in the set should NOT be null!",
            val );
        String newVal = val + "-not real";
        spu.setValue( ek, newVal );
        spu.reset();
        assertEquals(
            "The existing value did not get reset.",
            val,
            getValue( ek ) );
    }
    
    
    public void testReset2()
    {
        String ek = getExistingKey();
        String val = getValue( ek );
        assertNotNull(
            "The existing value in the set should NOT be null!",
            val );
        spu.setValue( ek, null );
        spu.reset();
        assertEquals(
            "The existing value did not get reset.",
            val,
            getValue( ek ) );
    }
    
    
    public void testReset3()
    {
        String uk = getUniqueKey();
        String origVal = getValue( uk );
        assertNull(
            "The unique key was not null, but '"+origVal+"'.",
            origVal );
        spu.setValue( uk, "a" );
        spu.reset();
        String val = getValue( uk );
        assertNull(
            "The unique key did not get reset to null, but was set to '"+
            val+"'.",
            val );
    }
    
    
    public void testReset4()
    {
        String ek = getExistingKey();
        String val = getValue( ek );
        String newVal = val + "-not real";
        spu.setValue( ek, newVal );
        spu.reset();
        spu.setValue( ek, null );
        spu.reset();
        assertEquals(
            "The existing value did not get reset.",
            val,
            getValue( ek ) );
    }
    
    
    public void testReset5()
    {
        String ek = getExistingKey();
        String val = getValue( ek );
        String newVal = val + "-not real";
        spu.setValue( ek, newVal );
        spu.reset();
        spu.setValue( ek, newVal+"1" );
        spu.reset();
        assertEquals(
            "The existing key did not get reset.",
            val,
            getValue( ek ) );
    }
    
    
    public void testReset6()
    {
        String uk = getUniqueKey();
        spu.setValue( uk, "a" );
        spu.reset();
        spu.setValue( uk, "b" );
        spu.reset();
        assertNull(
            "The unique key did not get reset.",
            getValue( uk ) );
    }
    
    
    public void testReset7()
    {
        spu.reset();
    }
    
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    private String getValue( String key )
    {
        return System.getProperty( key );
    }
    
    
    private String getUniqueKey()
    {
        Enumeration enum = System.getProperties().keys();
        String key = "";
        if (enum.hasMoreElements())
        {
            key = (String)enum.nextElement();
        }
        int count = 0;
        String uk = key;
        while (getValue( uk ) != null)
        {
            ++count;
            uk = key + count;
        }
        return uk;
    }
    
    
    private String getExistingKey()
    {
        Enumeration enum = System.getProperties().keys();
        if (enum.hasMoreElements())
        {
            return (String)enum.nextElement();
        }
        fail( "There are no existing system keys. Your JVM is wack." );
        // never reachable
        throw new IllegalStateException( "Not reachable." );
    }
    
    private SysPropertiesUtil spu;
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
        this.spu = new SysPropertiesUtil();
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        this.spu.reset();
        
        super.tearDown();
    }
}

