/*
 * @(#)AbstractSingleStore.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.util.classes.v1;



/**
 * Aids pluggable factories and related classes by being a central repository
 * for storing a singleton, and creating means to load and change the singleton.
 *
 * @author  Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version $Date: 2003/02/10 22:52:36 $
 * @since   March 30, 2002
 */
public abstract class AbstractSingleStore
{
    private Class instanceOf;
    private Object singleton;
    
    
    /**
     * Constructor specifying all the parameters for using a singleton in this
     * framework.
     *
     * @param instanceOf singletons must be of this class.
     */
    public AbstractSingleStore( Class instanceOf )
    {
        this.instanceOf = instanceOf;
    }
    
    
    /**
     * Returns the current inner singleton.  If it has never been set, then
     * the default will be used instead.
     *
     * @return the inner singleton instance.
     * @exception IllegalStateException if no singleton was created.
     */
    public Object getSingleton()
    {
        synchronized( this )
        {
            if (this.singleton == null)
            {
                setDefaultSingleton();
                if (this.singleton == null)
                {
                    throw new IllegalStateException( "No singleton created." );
                }
            }
        }
        return this.singleton;
    }
    
    
    /**
     * Sets the singleton.  It must be of the correct class, and non-null.
     *
     * @param singleton the singleton to set.
     * @exception IllegalArgumentException if <tt>singleton</tt> is
     *      <tt>null</tt>, or is not of the correct type.
     */
    public synchronized void setSingleton( Object singleton )
    {
        if (singleton == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        if (this.instanceOf != null &&
            !this.instanceOf.isInstance( singleton ))
        {
            throw new IllegalArgumentException( "Passed-in singleton "+
                singleton+" is not assignable to class "+
                this.instanceOf.getName()+", but is of class "+
                singleton.getClass().getName() );
        }
        this.singleton = singleton;
    }
    
    
    /**
     * Sets the inner singleton to the default, which is an implementation
     * specific method.
     */
    protected abstract void setDefaultSingleton();
    
    
    /**
     * Helper method to load an object from the class specified in the given
     * system property; if the class is invalid, then the given default
     * class will be used instead.  No cast testing is performed.
     *
     * @param key the System property to reference for the classname to
     *      instantiate.  It is passed to <tt>ClassLoadHelper</tt>.
     * @param defaultClass class to instantiate if the class defined in the
     *      system property is invalid.
     * @return the generated object.
     * @exception IllegalArgumentException if <tt>key</tt> is <tt>null</tt>.
     * @see ClassLoadHelper#createObjectFromProperty( String, Class, boolean )
     */
    protected static Object createFromProperty( String key,
            Class defaultClass )
    {
        if (key == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        ClassLoadHelper clh = new ClassLoadHelper();
        return clh.createObjectFromProperty( key, defaultClass, false );
    }
}

