/*
 * @(#)IChainableExceptionUTestI.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.util.throwable.v1;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.ByteArrayOutputStream;
import org.easymock.EasyMock;
import org.easymock.MockControl;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.junit.v1.iftc.*;


/**
 * Tests the IChainableException interface.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2003/05/06 05:35:02 $
 * @since     March 17, 2002
 */
public class IChainableExceptionUTestI extends InterfaceTestCase
{
    
    //-------------------------------------------------------------------------
    // Due to the way exceptions work, most of the logic is involved in the
    // constructor.  Hence, we'll use an alternative pattern to the standard
    // ImplFactory, where it will return a factory (below) that will create
    // exceptions using the correct constructor.
    
    public static interface IChainableExceptionFactory
    {
        public IChainableException createException();
        public IChainableException createException( String message );
        public IChainableException createException( Throwable cause );
        public IChainableException createException( String message,
            Throwable cause );
        public IChainableException createException( Throwable cause,
            String message );
    }
    
    
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS =
        IChainableExceptionUTestI.class;
    
    public IChainableExceptionUTestI( String name, ImplFactory f )
    {
        super( name, IChainableExceptionFactory.class, f );
    }

    
    //-------------------------------------------------------------------------
    // setup
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    private IChainableExceptionFactory createChainableExceptionFactory()
    {
        return (IChainableExceptionFactory)createImplObject();
    }
    
    
    protected IChainableException createException()
    {
        IChainableException ce = createChainableExceptionFactory().
            createException();
        assertIsRealChainableException( ce );
        return ce;
    }
    
    
    protected IChainableException createException( String message )
    {
        IChainableException ce = createChainableExceptionFactory().
            createException( message );
        assertIsRealChainableException( ce );
        return ce;
    }
    
    
    protected IChainableException createException( Throwable cause )
    {
        IChainableException ce = createChainableExceptionFactory().
            createException( cause );
        assertIsRealChainableException( ce );
        return ce;
    }
    
    
    protected IChainableException createException( String message,
            Throwable cause )
    {
        IChainableException ce = createChainableExceptionFactory().
            createException( message, cause );
        assertIsRealChainableException( ce );
        return ce;
    }
    
    
    protected IChainableException createException( Throwable cause,
            String message )
    {
        IChainableException ce = createChainableExceptionFactory().
            createException( cause, message );
        assertIsRealChainableException( ce );
        return ce;
    }
    
    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testEmptyConstructor1()
    {
        IChainableException ce = createException();
        assertNull(
            "Cause is not null.",
            ce.getCause() );
    }
    
    
    public void testEmptyConstructor2()
    {
        IChainableException ce = createException();
        Throwable t = new Throwable();
        ce.initCause( t );
        assertEquals(
            "Cause is not right.",
            t,
            ce.getCause() );
    }
    
    
    public void testEmptyConstructor3()
    {
        IChainableException ce = createException();
        try
        {
            ce.initCause( (Throwable)ce );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // check exception
        }
    }
    
    
    public void testCauseConstructor1()
    {
        Throwable t = null;
        IChainableException ce = createException( t );
        assertNull(
            "Cause is not null.",
            ce.getCause() );
    }
    
    
    public void testCauseConstructor2()
    {
        Throwable t = new Throwable();
        IChainableException ce = createException( t );
        assertEquals(
            "Cause is not right.",
            t,
            ce.getCause() );
    }
    
    
    public void testCauseConstructor3()
    {
        Throwable t = new Throwable();
        IChainableException ce = createException( t );
        try
        {
            ce.initCause( t );
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException ise)
        {
            // check exception
        }
    }
    
    
    public void testCauseConstructor4()
    {
        IChainableException ce = createException( (Throwable)null );
        try
        {
            ce.initCause( null );
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException ise)
        {
            // check exception
        }
    }
    
    
    public void testCauseConstructor5()
    {
        IChainableException ce = createException( (Throwable)null );
        try
        {
            ce.initCause( new Throwable() );
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException ise)
        {
            // check exception
        }
    }
    
    
    public void testCauseConstructor6()
    {
        IChainableException ce = createException( new Throwable() );
        try
        {
            ce.initCause( null );
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException ise)
        {
            // check exception
        }
    }
    
    
    public void testCauseConstructor7()
    {
        IChainableException ce = createException( new Throwable() );
        try
        {
            // order is important here - already set the throwable, so that
            // will be thrown first.
            ce.initCause( (Throwable)ce );
            fail( "Did not throw IllegalStateException." );
        }
        catch (IllegalStateException iae)
        {
            // check exception
        }
    }
    
    
    
    
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    private void assertIsRealChainableException( IChainableException ce )
    {
        assertNotNull( ce );
        assertTrue(
            "Class under test ("+ce.getClass()+") is not an exception.",
            ce instanceof Throwable );
    }
    
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static InterfaceTestSuite suite()
    {
        InterfaceTestSuite suite = new InterfaceTestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

